unit MainU;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, ToolWin, ComCtrls, SBPro, ImgList, ExtCtrls, GR32_Image,
  StdCtrls, Buttons, ExtDlgs, GR32, GR32_Layers;

type
  TMainFrm = class(TForm)
    gIn: TImgView32;
    Panel: TPanel;
    gOut: TImage32;
    btnOpen: TSpeedButton;
    btnSave: TSpeedButton;
    OpenDialog: TOpenPictureDialog;
    SaveDialog: TSaveDialog;
    shpBackClr: TShape;
    Label1: TLabel;
    ColorDialog: TColorDialog;
    btnCopy: TSpeedButton;
    rbtnTopLeft: TRadioButton;
    pnlAlign: TPanel;
    rbtnBottomLeft: TRadioButton;
    rbtnBottomRight: TRadioButton;
    rbtnTopRight: TRadioButton;
    rbtnTop: TRadioButton;
    rbtnLeft: TRadioButton;
    rbtnRight: TRadioButton;
    rbtnBottom: TRadioButton;
    edtAlignSize: TEdit;
    btnAlign: TButton;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure shpBackClrMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormCreate(Sender: TObject);
    procedure gInMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer; Layer: TCustomLayer);
    procedure btnOpenClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure btnCopyClick(Sender: TObject);
    procedure btnAlignClick(Sender: TObject);
    procedure rbtnClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    FPicture: TBitmap32;
    FMask: TBitmap32;
    FAlignBits: Integer;
    procedure OpenPic(const AFileName: String);
    procedure SavePic(const AFileName: String);
    procedure GetDrawingPiece(const Place: TPoint);
    procedure UpdatePreview;
  public
    { Public declarations }
  end;

var
  MainFrm: TMainFrm;

//==============================================================================

implementation
{$R *.dfm}

uses PNGImage, Math, Clipbrd;

//------------------------------------------------------------------------------

constructor TMainFrm.Create(AOwner: TComponent);
begin
  FPicture:= TBitmap32.Create;
  FMask:= TBitmap32.Create;
  inherited Create(AOwner);
end;

//------------------------------------------------------------------------------

destructor TMainFrm.Destroy;
begin
  FPicture.Free;
  FMask.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.OpenPic(const AFileName: String);
// universal loader
var Src: TPicture;
begin
  if FileExists(AFileName) then begin
    Src:= TPicture.Create;
    Src.LoadFromFile(AFileName);
    gIn.Bitmap.Assign(Src);
    FMask.SetSizeFrom(gIn.Bitmap);
    Src.Free;
  end else raise Exception.CreateFmt('File %s does not exit',[AFileName]);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.SavePic(const AFileName: String);
begin
  with TPNGObject.Create do begin
    Assign(FPicture);
    CompressionLevel:= High(TCompressionLevel); // or 9
    SaveToFile(AFileName);
    Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.shpBackClrMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  with ColorDialog, shpBackClr do begin
    Color:= Brush.Color;
    if (Button = mbLeft) and Execute then Brush.Color:= Color;
  end;  
end;

//------------------------------------------------------------------------------

procedure TMainFrm.FormCreate(Sender: TObject);
begin
  shpBackClr.Brush.Color:= $00FFFFE7; // simu back in windows notation ;-)
  FMask.SetSizeFrom(gIn.Bitmap);
  with gOut.Bitmap do begin
    SetSize(132, 86);
    Clear(Color32(clBtnFace));
    FrameRectS(0, 0, 131, 85, Color32(clWindowText));
    RenderText(2, 2, 'preview empty', 0, Color32(clWindowText));
  end;
  FAlignBits:= 1;
  if ParamCount = 1 then OpenPic(ParamStr(1));
end;

//------------------------------------------------------------------------------

procedure TMainFrm.gInMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer; Layer: TCustomLayer);
var pos: TPoint;
begin
  pos.X:= X; pos.Y:= Y;
  pos:= gIn.ControlToBitmap(pos);
  // grab click position and convert
  case Button of
    mbRight, mbMiddle: with gIn do begin // zoom
      ScrollToCenter(pos.X, pos.Y);
      // center the picture around clicked point
    end;
    mbLeft: GetDrawingPiece(pos);
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.GetDrawingPiece(const Place: TPoint);
var PieceClr, BackClr, ToDoClr: TColor32;
    MaxX, MaxY,
    X, Y,
    LenX, LenY,
    CntX, CntY,
    ValX, ValY,
    HiX, HiY, LoX, LoY: Integer;
    SrchRadius, i, StackPos: Integer;
    bmpPict: TBitmap32;
    BPict, BMask: PColor32Array;
    PieceBounds: TRect;
    Continue: Boolean;
    (* explanation of X,Y variables:
      Max? - maximal possible
      Len? - size (Max + 1)
      Cnt? - counter for for 2nd level loops
      Val? - backup for finding loops
      Hi?, Lo? - 2nd level loops (for i:= Lo? to Hi?)
      ? - position for next loop iteration

      Other important vars:
      SrchRadius - maximal distance to search in
      i - main finding (square) loop counter
      StackPos - X,Y converted into one dimensional array position
    *)
  //............................................................................
  procedure Check; 
  begin
    StackPos:= (Y * LenX) + X;
    if (BPict[StackPos] <> BackClr) and (BMask[StackPos] = BackClr) then
        BMask[StackPos]:= ToDoClr // should be processed
  end;
  //............................................................................
begin
  bmpPict:= gIn.Bitmap;
  BPict:= bmpPict.Bits;
  BMask:= FMask.Bits;
  // "aliases" - let's have pointers for inner parts instead of WITHs etc. 

  BackClr:= Color32(shpBackClr.Brush.Color) or $FF000000;
  // background is the same color as set in GUI and _opaque_
  PieceClr:= BackClr and $00FFFFFF;
  // piece is fully transparent and regardless what color, so let's use the same 
  ToDoClr:= $00FFFFFF and not PieceClr;
  (* to do must be :
     1) different from both other ones - they have the same RGB, so inversion
        grants sth different
     2) fully transparent, because that's the border around which separates
     "pieces"
  *)

  with Place, bmpPict do Continue:=
    (PixelS[X, Y] <> BackClr)
    and PtInRect(BoundsRect, Place);
  if not Continue then raise EAbort.Create('Nothing there.');

  PieceBounds:= bmpPict.BoundsRect;
  with PieceBounds do begin
    i:= Bottom; // i "hijacked" for swapping variables
    Bottom:= Top;
    Top:= i;
    i:= Left;
    Left:= Right;
    Right:= i;
  end; 
  with bmpPict do begin
    LenX:= Width;
    LenY:= Height;
    MaxX:= LenX - 1;
    MaxY:= LenY - 1;
  end;
  // "localize" variables - loop is time extensive, every bit of speed is
  // probably needed.
  FMask.DrawMode:= dmBlend;
  FMask.Clear(BackClr);
  // clear mask
  X:= Place.X;
  Y:= Place.Y;
  // set the starting point
  // * initialization done


  while X <> -1 do begin
    
    BMask[(Y * LenX) + X]:= PieceClr; // set to already visited and processed
    with PieceBounds do begin
      if Top > Y then Top:= Y;
      if Bottom < Y then Bottom:= Y;
      if Right < X then Right:= X;
      if Left > X then Left:= X;
    end;
    // update detection bounding rectangle

    // Now check the 8 surrounding pixels
    // first the 4 "straight" ones
    inc(X); // X + 1
    if (X < LenX) then Check;
    // If it's not outside, try to mark it as another checkpoint.
    dec(X); dec(X); // X - 1
    if (X > -1) then Check;
    inc(X); // restore X to original value
    inc(Y); // Y + 1
    if (Y < LenY) then Check;
    dec(Y); dec(Y); // Y - 1
    if (Y > -1) then Check;
    inc(Y); // restore Y to original value
    // and the 4 diagonally around.  
    inc(X);
    inc(Y); // X + 1, Y + 1
    if (X < LenX) and (Y < LenY) then Check;
    dec(Y); dec(Y); // X + 1, Y - 1
    if (X < LenX) and (Y > -1) then Check;
    dec(X); dec(X); // X - 1, Y - 1
    if (X > -1) and (Y > -1) then Check;
    inc(Y); inc(Y); // X - 1, Y + 1
    if (X > -1) and (Y < LenY) then Check;  // (Y < LenX) ????
    inc(X);
    dec(Y); // back to original values
    // check of surroundings finished, now look for next "workpoint"
    
    ValX:= X;
    ValY:= Y;
    X:= -1;
    // Invalid value, will cause loop end; is changed when next point is found.
    // If no point is found and -1 stays, work is over.
    SrchRadius:= MaxIntValue([ValX, ValY, MaXX - ValX, MaxY - ValY]);
    // the largest distance between "workpoint" and picture border

    for i:= 1 to SrchRadius do begin // search incrementally in squares around  
      // a square is formed by 4 lines - 4 loops

      if X = -1 then begin // if nothing was found (redundant for 1st loop) ...
        // Top: Left -> Right
        CntY:= ValY - i; // set one coordinate as fixed
        if CntY > -1 then begin // only if line intersects with picture
          LoX:= ValX - i; 
          HiX:= ValX + i; // calculate start and end of line
          if LoX < 0 then LoX:= 0;
          if HiX > MaxX then HiX:= MaxX; // crop parts outside picture
          for CntX:= LoX to HiX do if BMask[(CntY * LenX) + CntX] = ToDoClr then begin
            // iterate along line and search for 1
            X:= CntX;
            Y:= CntY;                              
            Break; 
          end;
        end;  
      end;  

      if X = -1 then begin // ^ ... but must be here
        // Right: Top -> Bottom
        CntX:= ValX + i;
        if CntX < LenX then begin
          LoY:= ValY - i;
          HiY:= ValY + i;
          if LoY < 0 then LoY:= 0;
          if HiY > MaxY then HiY:= MaxY; 
          for CntY:= LoY to HiY do if BMask[(CntY * LenX) + CntX] = ToDoClr then begin
            X:= CntX;
            Y:= CntY;
            Break;
          end;
        end;  
      end;

      if X = -1 then begin
        // Bottom: Right -> Left
        CntY:= ValY + i;
        if CntY < LenY then begin
          HiX:= ValX + i;
          LoX:= ValX - i;
          if LoX < 0 then LoX:= 0;
          if HiX > MaxX then HiX:= MaxX;
          for CntX:= LoX to HiX do if BMask[(CntY * LenX) + CntX] = ToDoClr then begin
            X:= CntX;
            Y:= CntY;
            Break;
          end;
        end;  
      end;

      if X = -1 then begin
        // Left: Bottom -> Top
        CntX:= ValX - i;
        if CntX > -1 then begin
          HiY:= ValY + i;
          LoY:= ValY - i;
          if LoY < 0 then LoY:= 0;
          if HiY > MaxY then HiY:= MaxY; 
          for CntY:= LoY to HiY do if BMask[(CntY * LenX) + CntX] = ToDoClr then begin
            X:= CntX;
            Y:= CntY;
            Break;
          end;
        end;  
      end;

      if X <> -1 then Break; // If some of the inner loops made a hit, cancel
      // the enclosing square-iterating loop, too. 
    end;
    
    // If nothing was found, floodfill is finished; X stays -1 and thus main 
    // loop stops.
  end;
  
  with PieceBounds, FPicture do begin
    Inc(Bottom);
    Inc(Right);
    // final cosmetics on bounding rectangle
    SetSize(Abs(Right - Left), Abs(Bottom - Top));
    Draw(0, 0, PieceBounds, bmpPict); // put the part of picture there
    Draw(0, 0, PieceBounds, FMask); // and mask out unwanted parts
  end;

  UpdatePreview;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.btnOpenClick(Sender: TObject);
begin
  with OpenDialog do if Execute then OpenPic(FileName);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.btnSaveClick(Sender: TObject);
begin
  with SaveDialog do if Execute then SavePic(FileName);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.btnCopyClick(Sender: TObject);
var P: TPicture;
begin
  P:= TPicture.Create;
  P.Assign(FPicture);
  Clipboard.Assign(P);
  P.Free;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.UpdatePreview;
var DrawScale: Real;
    Src, Dst: TRect;
begin
  DrawScale:= Min(gOut.Width / FPicture.Width, gOut.Height / FPicture.Height);
  Src:= FPicture.BoundsRect;
  Dst:= Src;
  with Dst do begin
    Bottom:= Floor(DrawScale * Bottom) + 1;
    Right:= Floor(DrawScale * Right) + 1; 
  end;  
  with gOut.Bitmap do begin
    Clear(Color32(clBtnFace)); // shpBackClr.Brush.Color
    Draw(Dst, Src, FPicture);
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.btnAlignClick(Sender: TObject);
var NewH, NewW: Integer;
    OldH, OldW: Integer;
    RestH, RestW: Integer;
    PosH, PosW: Integer;
    AlignSize: Integer;
    FBuffer: TBitmap32;
(*
   strange numbers = direction bits:

   1 or 2 = 3  -----------  1  -------------  1 or 8 = 9

   |         A                                         |
   |         |                                         |
             
   2     1 or 4 = 5         <-- 8 or 2 = 10 -->        8                           
         
   |         |                                         |
   |         V                                         |

   4 or 2 = 6  ------------  4  -----------  4 or 8 = 12
*)
begin 
  if FPicture.Empty then raise EAbort.Create('FPicture empty.');
  AlignSize:= StrToInt(edtAlignSize.Text);
  AlignSize:= EnsureRange(Abs(AlignSize), 16, 254);
  // exception here if not a number
  FBuffer:= TBitmap32.Create;
  FBuffer.Assign(FPicture);
  // prepare backup
  with FPicture do begin
    OldH:= Height;
    OldW:= Width;
  end;
  // get old size
  if (FAlignBits and 5) <> 0 then begin
    RestH:= OldH mod AlignSize;
    if RestH > 0 then NewH:= OldH + (AlignSize - RestH) else NewH:= OldH;
  end else NewH:= OldH;
  if (FAlignBits and 10) <> 0 then begin
    RestW:= OldW mod AlignSize;
    if RestW > 0 then NewW:= OldW + (AlignSize - RestW) else NewW:= OldW;
  end else NewW:= OldW;
  // Find new size if the picture is to be resized in the direction. Note that
  // the rest can be 0, which means it's already aligned, but using formula
  // (AlignSize - Rest) would result in widening the picture by full AlignSize. 
  with FPicture do begin
    SetSize(NewW, NewH);
    Clear(Color32(shpBackClr.Brush.Color));
  end;
  if (FAlignBits and 5) = 4 then PosH:= NewH - OldH else PosH:= 0;
  if (FAlignBits and 10) = 8 then PosW:= NewW - OldW else PosW:= 0;
  (*case FAlignBits and 5 of
    1, 0: PosH:= 0;
    4: PosH:= NewH - OldH;
  end;   
  case FAlignBits and 10 of
    2, 0: PosW:= 0;
    8: PosW:= NewW - OldW;
  end;
  *)
  FPicture.Draw(PosW, PosH, FBuffer);
  FBuffer.Free;
  UpdatePreview;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.rbtnClick(Sender: TObject);
begin
  FAlignBits:= TRadioButton(Sender).Tag;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  case Key of
    Ord('O'): btnOpen.Click;
    Ord('S'): btnSave.Click;
    Ord('C'): btnCopy.Click;
  end;
end;

end.
