unit ColorU;
                 
//==============================================================================
    INTERFACE
//==============================================================================

uses SysUtils, Classes, Types, Graphics, GR32;

//------------------------------------------------------------------------------

type TColorByteRec = packed record case Integer of
       0: (Color: TColor;);
       1: (Red:   Byte;
           Green: Byte;
           Blue:  Byte;
           Alpha: Byte;);
     end;

//------------------------------------------------------------------------------

type EColorConvertException = class(Exception);


function HexToColor(const Hex: String): TColor;

function ColorToHex(const Color: TColor): String;

function SingleHexToByte(const Hex: Char): Byte;

function ByteToSingleHex(const B: Byte): Char;

function DoubleHexToByte(const Hex: String): Byte;

function ByteToDoubleHex(const B: Byte): String;

function HexToColorDef(const Hex: String; Default: TColor): TColor;

function Cl32ToHex(const Color: TColor32): String;

function Cl32ToHexPlain(const Color: TColor32): String;

function Cl32ToDec(const Color: TColor32): String;

function Cl32ToDecPlain(const Color: TColor32): String;

function HexToCl32(const Hex: String): TColor32;

function HexToCl32Def(const Hex: String; const Default: TColor32): TColor32;

function Cl32ToHexShort(const Color: TColor32): String;

//------------------------------------------------------------------------------

type TColor32Metric = function(const C1, C2: TColor32): Real;


function ColorDistanceRGBEuclid2(const C1, C2: TColor32): Real;

function ColorDistanceRGBWeighted2(const C1, C2: TColor32): Real;

function ColorDistanceManhattan(const C1, C2: TColor32): Real;

//------------------------------------------------------------------------------

function GrayAvg(const Color: TColor32): TColor32;

function GrayLum(const Color: TColor32): TColor32;

//==============================================================================

IMPLEMENTATION

uses ConvOrdU;

//------------------------------------------------------------------------------

const ClConvData: array [$00 .. $0F] of Char =
      ('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D',
        'E', 'F');

//==============================================================================
// color <-> hex routines

function SingleHexToByte(const Hex: Char): Byte;
begin
  case Hex of
    '0'..'9': Result:= Ord(Hex) - 48;
    'A'..'F': Result:= Ord(Hex) - 55; // -65+10
    'a'..'f': Result:= Ord(Hex) - 87; // -97+10
    else raise EColorConvertException.Create('ColorU.SingleHexToByte:'#10 +
      Hex + ' is not a valid hexadecimal digit.');
  end;
end;

//------------------------------------------------------------------------------

function DoubleHexToByte(const Hex: String): Byte;
begin
  case Length(Hex) of
    1: Result:= SingleHexToByte(Hex[1]);
    2: Result:= SingleHexToByte(Hex[1]) * 16 + SingleHexToByte(Hex[2]);
    else raise EColorConvertException.Create('ColorU.DoubleHexToByte:' + #10 +
      Hex + ' is not a valid two-digit hexadecimal value.');
  end; 
end;

//------------------------------------------------------------------------------

function HexToColor(const Hex: String): TColor;
var S: String;
    C: TColorByteRec;
    I: Integer;
begin
  S:= Hex; //S:= UpperCase(Hex);
  case Length(S) of
    6: ; //no correction needed
    3: S:= S[1] + S[1] + S[2] + S[2] + S[3] + S[3]; // rare but possible to encounter
    //2: S:= S+S+S;
    //1: S:= S+S+S+S+S+S;
    // not useful, these formats ... useless 
    else raise EColorConvertException.Create(
      'Cannot resolve hexadecimal color representation.');
     // not valid combination
  end;
  for I:= 1 to 6 do if not (S[I] in ['A'..'F', '0'..'9', 'a'..'f']) then raise
    EColorConvertException.Create('ColorU.HexToColor:'#10'  ' + Hex +
    ' is not a valid hexadecimal value.');
  // correction & detection finished, now process every char
  C.Alpha:= 0;
  C.Red:= SingleHexToByte(S[1]) * 16 + SingleHexToByte(S[2]);
  C.Green:= SingleHexToByte(S[3]) * 16 + SingleHexToByte(S[4]);
  C.Blue:= SingleHexToByte(S[5]) * 16 + SingleHexToByte(S[6]);
  // 16* upper digit + lower digit ; for every color part
  Result:= C.Color;
end;

//------------------------------------------------------------------------------

function ByteToSingleHex(const B: Byte): Char;
begin
  case B of
    0 .. 9: Result:= Chr(B + 48);
    10 .. 15: Result:= Chr(B + 55);
    else raise EColorConvertException.Create(InttoStr(B) +
      ' is not a valid number for conversion to hex digit.');
  end;
end;

//------------------------------------------------------------------------------

function ByteToDoubleHex(const B: Byte): String;
begin
 Result:= ByteToSingleHex(B div 16) + ByteToSingleHex(B mod 16);
end;

//------------------------------------------------------------------------------

function ColorToHex(const Color: TColor): String;
var C: TColorByteRec;
    S: String;
    B: Byte;
begin
  SetLength(S, 6);
  C.Color:= Color;
  B:= C.Red div 16;
  S[1]:= ByteToSingleHex(B);
  B:= C.Red mod 16;
  S[2]:= ByteToSingleHex(B);
  B:= C.Green div 16;
  S[3]:= ByteToSingleHex(B);
  B:= C.Green mod 16;
  S[4]:= ByteToSingleHex(B);
  B:= C.Blue div 16;
  S[5]:= ByteToSingleHex(B);
  B:= C.Blue mod 16;
  S[6]:= ByteToSingleHex(B);
  Result:= S;
end;

//------------------------------------------------------------------------------

function HexToColorDef(const Hex: String; Default: TColor): TColor;
var Value: TColor;
begin
  Value:= Default;
  try
    Value:= HexToColor(Hex);
  finally
    Result:= Value;
  end;
end;

//------------------------------------------------------------------------------

function Cl32ToHex(const Color: TColor32): String;
var B: TRec4;
begin
  SetLength(Result, 8);
  B.fCardinal:= Color;
  Result[1]:= ClConvData[B.fByte[3] shr 4];
  Result[2]:= ClConvData[B.fByte[3] and $0F];
  Result[3]:= ClConvData[B.fByte[2] shr 4];
  Result[4]:= ClConvData[B.fByte[2] and $0F];
  Result[5]:= ClConvData[B.fByte[1] shr 4];
  Result[6]:= ClConvData[B.fByte[1] and $0F];
  Result[7]:= ClConvData[B.fByte[0] shr 4];
  Result[8]:= ClConvData[B.fByte[0] and $0F];
end;

//------------------------------------------------------------------------------

function HexToCl32(const Hex: String): TColor32;
var B: TRec4;
begin
  Result:= clBlack32;
  if Length(Hex) = 8 then begin
    B.fByte[3]:= DoubleHexToByte(Copy(Hex, 1, 2));
    B.fByte[2]:= DoubleHexToByte(Copy(Hex, 3, 2));
    B.fByte[1]:= DoubleHexToByte(Copy(Hex, 5, 2));
    B.fByte[0]:= DoubleHexToByte(Copy(Hex, 7, 2));
    Result:= B.fCardinal;
  end else raise EColorConvertException.CreateFmt(
    'String "%s" is not a valid hexadecimal 4-byte number', [Hex]);
end;

//------------------------------------------------------------------------------

function HexToCl32Def(const Hex: String; const Default: TColor32): TColor32;
var Value: TColor32;
begin
  Value:= Default;
  try
    Value:= HexToCl32(Hex);
  finally
    Result:= Value;
  end;
end;

//------------------------------------------------------------------------------

function Cl32ToHexShort(const Color: TColor32): String;
var B: TRec4;
begin
  Result:= EmptyStr;
  SetLength(Result, 6);
  B.fCardinal:= Color;
  Result[1]:= ClConvData[B.fByte[2] shr 4];
  Result[2]:= ClConvData[B.fByte[2] and $0F];
  Result[3]:= ClConvData[B.fByte[1] shr 4];
  Result[4]:= ClConvData[B.fByte[1] and $0F];
  Result[5]:= ClConvData[B.fByte[0] shr 4];
  Result[6]:= ClConvData[B.fByte[0] and $0F];
end;

//------------------------------------------------------------------------------

function Cl32ToDec(const Color: TColor32): String;
begin
  Result:= EmptyStr;
  Result:= IntToStr((Color shr 24) and $FF) +
     ',' + IntToStr((Color shr 16) and $FF) +
     ',' + IntToStr((Color shr 8) and $FF) +
     ',' + IntToStr(Color and $FF);
end;

//------------------------------------------------------------------------------

function Cl32ToDecPlain(const Color: TColor32): String;
begin
  Result:= EmptyStr;
  Result:= IntToStr((Color shr 16) and $FF) +
     ',' + IntToStr((Color shr 8) and $FF) +
     ',' + IntToStr(Color and $FF);
end;

//------------------------------------------------------------------------------

function Cl32ToHexPlain(const Color: TColor32): String;
var B: TRec4;
begin
  Result:= EmptyStr;
  SetLength(Result, 6);
  B.fCardinal:= Color;
  Result[1]:= ClConvData[B.fByte[2] shr 4];
  Result[2]:= ClConvData[B.fByte[2] and $0F];
  Result[3]:= ClConvData[B.fByte[1] shr 4];
  Result[4]:= ClConvData[B.fByte[1] and $0F];
  Result[5]:= ClConvData[B.fByte[0] shr 4];
  Result[6]:= ClConvData[B.fByte[0] and $0F];
end;

//==============================================================================

function ColorDistanceRGBEuclid2(const C1, C2: TColor32): Real;
begin
  if C1 = C2 then
    Result:= 0.0
  else
    Result:=
    Sqr(TColor32Entry(C1).R - TColor32Entry(C2).R) +
    Sqr(TColor32Entry(C1).G - TColor32Entry(C2).G) +
    Sqr(TColor32Entry(C1).B - TColor32Entry(C2).B)
    ;
end;

//------------------------------------------------------------------------------
// see:
// http://www.compuphase.com/cmetric.htm  @ paragraph "A low-cost approximation"

function ColorDistanceRGBWeighted2(const C1, C2: TColor32): Real;
var R_mean: Integer;
begin
  if C1 = C2 then
    Result:= 0.0
  else begin
    R_mean:= (TColor32Entry(C1).R + TColor32Entry(C2).R) div 2;
    Result:=
      (Sqr(TColor32Entry(C1).R - TColor32Entry(C2).R) * (512 + R_mean)) shr 8 + // div 256
      Sqr(TColor32Entry(C1).G - TColor32Entry(C2).G) shl 2 + // *4
      (Sqr(TColor32Entry(C1).B - TColor32Entry(C2).B) * (767 - R_mean)) shr 8 // div 256
    ;
  end;
end;

//------------------------------------------------------------------------------

function ColorDistanceManhattan(const C1, C2: TColor32): Real;
begin
  Result:=
    Abs(TColor32Entry(C1).R - TColor32Entry(C2).R) +
    Abs(TColor32Entry(C1).G - TColor32Entry(C2).G) +
    Abs(TColor32Entry(C1).B - TColor32Entry(C2).B)
  ;
end;

//------------------------------------------------------------------------------

function GrayAvg(const Color: TColor32): TColor32;
begin
  Result:= Gray32(Round((
    (Color and $FF) + (Color shr 8 and $FF) + (Color shr 16 and $FF)
    ) / 3.0));
end;

//------------------------------------------------------------------------------

function GrayLum(const Color: TColor32): TColor32;
var Value: Cardinal;
begin
  Value:= (11 * (Color shr 16 and $FF) + (Color and $FF00 shr 4) + 5 * (Color and $FF)) shr 5;
  // (11 * Red  +  16 * Green  +  5 * Blue) / 32
  Result:= Gray32(Value);
end;

//==============================================================================

END.
