unit FileU;

interface

uses SysUtils, Classes, StrU;

//------------------------------------------------------------------------------

procedure FindFilesByMask(const Spec: String; const Files: TStrings);

procedure FindFolders(const Spec: String; const Folders: TStrings;
    const AHidden: Boolean = True;
    const ASysFile: Boolean = True;
    const AReadOnly: Boolean = True;
    const AArchive: Boolean = True);

procedure FindFiles(const Spec: String; const Files: TStrings;
    const AHidden: Boolean = True;
    const ASysFile: Boolean = True;
    const AReadOnly: Boolean = True;
    const AArchive: Boolean = True);

function ForceDirTree(const Tree: String): Boolean;

function IsFolder(const Target: String): Boolean;

function SizeOfFile(const FileName: String): Int64;

function IsDotFolder(const Path: String): Boolean;

//==============================================================================
implementation

uses Math;
//==============================================================================

procedure FindFilesByMask(const Spec: String; const Files: TStrings);
var Sr: TSearchRec;
    FileAttrs: Integer;
begin
  FileAttrs:= faReadOnly or faArchive or faSysFile or faHidden;
  if FindFirst(Spec, FileAttrs, Sr) = 0 then repeat
    Files.Add(Sr.Name);
  until FindNext(Sr) <> 0;
  FindClose(Sr);
end;

//------------------------------------------------------------------------------

function ForceDirTree(const Tree: String): Boolean;
var Folders: TStrings;
    i: Integer;
    CurDir: String; 
begin
  Folders:= TStringList.Create;
  Result:= False;
  CurDir:= EmptyStr;
  try
    SplitString(GoodSlashes(Tree), PathDelim, Folders);
    with Folders do if Count > 0 then begin
      if (Length(Strings[0]) = 2) and (Strings[0][2] = DriveDelim) then begin
        CurDir:= Strings[0];
        Delete(0);
      end;
      // if first is drive, keep it separately and delete that item
      if Count = 0 then begin
        Result:= True;
        Abort;
      end;
      // if the # is now 0, there was only drive -> success :-)   
      for i:= 0 to Count - 1 do begin
        CurDir:= CurDir + PathDelim + Strings[i];
        Result:= DirectoryExists(CurDir) or CreateDir(CurDir);
        if not Result then Abort;
      end;
    end;
  finally
    Folders.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure FindFolders(const Spec: String; const Folders: TStrings;
    const AHidden: Boolean = True;
    const ASysFile: Boolean = True;
    const AReadOnly: Boolean = True;
    const AArchive: Boolean = True);
var Sr: TSearchRec;
    FileAttrs: Integer;
begin
  FileAttrs:= faDirectory
    or IfThen(AReadOnly, faReadOnly)
    or IfThen(AArchive, faArchive)
    or IfThen(ASysFile, faSysFile)
    or IfThen(AHidden, faHidden);
  if FindFirst(Spec, FileAttrs, Sr) = 0 then repeat
    if (SR.Attr and faDirectory) = faDirectory then Folders.Add(Sr.Name);
  until FindNext(Sr) <> 0;
  FindClose(Sr);
end;

//------------------------------------------------------------------------------

procedure FindFiles(const Spec: String; const Files: TStrings;
    const AHidden: Boolean = True;
    const ASysFile: Boolean = True;
    const AReadOnly: Boolean = True;
    const AArchive: Boolean = True);
var Sr: TSearchRec;
    FileAttrs: Integer;
begin
  FileAttrs:= 0
    or IfThen(AReadOnly, faReadOnly)
    or IfThen(AArchive, faArchive)
    or IfThen(ASysFile, faSysFile)
    or IfThen(AHidden, faHidden);
  if FindFirst(Spec, FileAttrs, Sr) = 0 then repeat
    if (SR.Attr and faDirectory) <> faDirectory then Files.Add(Sr.Name);
  until FindNext(Sr) <> 0;
  FindClose(Sr);
end;

//------------------------------------------------------------------------------

function IsFolder(const Target: String): Boolean;
const FolderAttrs = faReadOnly or faArchive or faSysFile or faHidden or faDirectory;
var s: TSearchRec;
    t: String;
    temp: TStringList;
begin
  temp:= TStringList.Create;
  t:= WinBkSlashes(Target);
  if t[Length(t)] = '\' then t:= Copy(t, 1, Length(t) - 1);
  SplitString(t, '\', temp);
  with temp do Result:=
    (IndexOf('.') = -1)
    and (IndexOf('..') = -1)
    and (AnsiPos('*', t) = 0)
    and (AnsiPos('?', t) = 0)
    and (FindFirst(t, FolderAttrs, s) = 0)
    and (s.Attr and faDirectory = faDirectory);
  FindClose(s);
  temp.Free;  
end;

//------------------------------------------------------------------------------

function IsDotFolder(const Path: String): Boolean;
begin
  Result:= (Path = '.') or (Path = '..');
end;

//------------------------------------------------------------------------------

function SizeOfFile(const FileName: String): Int64;
var F: TFileStream;
begin
  try
    try
      F:= TFileStream.Create(WinBkSlashes(FileName), fmOpenRead);
      Result:= F.Size;
    except on E: Exception do
      Result:= High(Int64); // give too big if inaccessible
    end;
  finally
    F.Free;
  end;
end;        

//------------------------------------------------------------------------------

end.
