unit RatMath;

interface

uses
  SysUtils,
  DynIntU,
  BasicDynIntProcs,
  RatNumU,
  Math,
  Classes,
  Dialogs;

function RIntPower(const X: TNumber; const Power: Integer): TTempNumber;

function RExp(const X: TNumber): TTempNumber;
function RLn(const X: TNumber; const Iters: Cardinal = 10): TTempNumber;

function RLog2(const X: TNumber; const Iters: Cardinal = 10): TTempNumber;
function RLog10(const X: TNumber; const Iters: Cardinal = 10): TTempNumber;

function RSin(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RCos(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RSec(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RCsc(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RTan(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RCot(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;

function RSqrt(const X: TNumber; const TrySquare: Boolean; const Iters: Cardinal = 5): TTempNumber;

function RPower(const Base, Power: TNumber; const Iters: Cardinal = 10): TTempNumber;

function RIntRoot(const X: TNumber; const Power: Integer; const Iters: Cardinal = 20): TTempNumber;

function RArcSin(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RArcCos(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RArcTan(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RArcCot(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RArcSec(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
function RArcCsc(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;

procedure CutTo(const Target: TNumber; const Bytes: Cardinal);

var RLn2, RLn10, RPi: TNumber;


implementation

var Loader: TFileStream;
    l: Integer;
    B: Byte;

procedure CutTo(const Target: TNumber; const Bytes: Cardinal);
var L: Cardinal;
begin
  L := Min(Target.Citatel.Length, Target.Jmenovatel.Length); // zjistime delku kratsiho
  if L > Bytes then begin // jestli se da posouvat tak posuneme
    BShr(Target.Citatel, (L-Bytes-1) shl 3); // nejprve jen na Bytes + 1
    BShr(Target.Jmenovatel, (L-Bytes-1) shl 3);
    if DIGet(Target.Citatel, 0) and 128 <> 0 then begin // a rozhodneme o zaokrouhleni
      BShr(Target.Citatel, 8);
      BInc(Target.Citatel);
    end else BShr(Target.Citatel, 8);
    if DIGet(Target.Jmenovatel, 0) and 128 <> 0 then begin // to same provedeme
      BShr(Target.Jmenovatel, 8);                          // se jmenovatelem
      BInc(Target.Jmenovatel);
    end else BShr(Target.Jmenovatel, 8);
  end;
end;

//------------------------------------------------------------------------------

function RIntPower(const X: TNumber; const Power: Integer): TTempNumber;
var CurPower: TNumber;
    AbsPower: Cardinal;
    Temp: TDynInt;
begin
  if not (BIsZero(X.Citatel) and (Power < 0)) then begin
    AbsPower := Abs(Power); // urcime abs. na co mame umocnit
    CurPower := TNumber.Create;
    CurPower.AssignValue(X);      // vytvorime objekty
    Result := TTempNumber.Create(true);
    if (AbsPower and 1) = 1 then begin
      Result.AssignValue(CurPower);
    end else begin            // rozhodneme jestli je exponent sudy nebo lichy
      Result.AssignValue(1);
    end;
    AbsPower := AbsPower shr 1; // posuneme exponent
    while AbsPower <> 0 do begin // dokud je na co umocnovat
      BMul(CurPower.Citatel, CurPower.Citatel, CurPower.Citatel); // CurPower se umocni na druho
      BMul(CurPower.Jmenovatel, CurPower.Jmenovatel, CurPower.Jmenovatel);
      if (AbsPower and 1) = 1 then AMul(Result, CurPower); // jestli je dana mocnina dvojky
      AbsPower := AbsPower shr 1; // v AbsPower, tak tim vynasobime Result
    end;                          // AbsPower se postupne posouva doprava
    if Power < 0 then begin // jestlize je exponent zaporny
      Temp := Result.Citatel; // tak prohodime citatele a jmenovatele
      Result.Citatel := Result.Jmenovatel;
      Result.Jmenovatel := Temp;
    end;
    CurPower.Free;
  end else raise EDivByZero.Create('division by zero');
end;

//------------------------------------------------------------------------------

function RExp(const X: TNumber): TTempNumber;
var i, Iters: Integer;
    Current, Pom: TNumber;
    Temp: TDynInt;
    Switch: Boolean;
begin
  Pom := TNumber.Create;
  Switch := X.Minus; // rozhodneme o prohozeni
  Pom.AssignValue(RAbs(X)); // a priradime absolutni hodnotu
  Current := TNumber.Create;
  Current.AssignValue(Pom);
  Result := TTempNumber.Create(true);
  Result.AssignValue(RAdd(Current, 1));
  Iters := Max(Pom.Int64Value*6, 20); // urcime pocet iteraci
  DICreate(Temp);
  for i := 2 to Iters do begin
    CutTo(Result, 10); // zkratime Result
    BMovInt64(Temp, i);
    CancelFrac(Current.Citatel, Temp);     // vygenerujeme novy clen rady
    BMul(Current.Jmenovatel, Current.Jmenovatel, Temp);
    Current.AssignValue(RMul([Current, Pom]));
    CutTo(Current, 10); // zkratime clen rady
    AAdd(Result, Current); // a pricteme k vysledku
  end;
  Current.Free;
  Pom.Free;     // zbavime se docasnych
  DIFree(Temp);
  if Switch then begin         // pro zaporne exponenty prohodime cit. a jm.
    Temp := Result.Citatel;
    Result.Citatel := Result.Jmenovatel;
    Result.Jmenovatel := Temp;
  end;
end;

//------------------------------------------------------------------------------

function RLn(const X: TNumber; const Iters: Cardinal = 10): TTempNumber;
var i: Integer;
    AM, GM, Tmp: TNumber;
begin
  AM := TNumber.Create;
  GM := TNumber.Create;
  Tmp := TNumber.Create;
  GM.AssignValue(MulInv(X));
  BShl(GM.Jmenovatel, 150); // posuneme argument
  if EndingZeros(GM.Jmenovatel) > 0 then CancelFrac(GM.Citatel, GM.Jmenovatel); // a pokratime
  AM.AssignValue(1);
  for i := 1 to Iters do begin
    CutTo(AM, 15); // zkratime AM a GM
    CutTo(GM, 15);
    Tmp.AssignValue(RMul([RAdd([AM, GM])], 1, 2)); // urcime aritmeticky prumer
    GM.AssignValue(RSqrt(RMul([GM, AM]), false, 10)); // a geometricky
    AM.AssignValue(Tmp);
  end;
  Tmp.AssignValue(MulInv(RMul([RAdd([AM, GM])]))); // secteme AM a GM, vysledkem je zhruba 2*AGM
  Result := RAdd([ RMul([RPi, Tmp]), AddInv(RMul(RLn2, 152)) ]); // urcime Ln
  Tmp.Free;
  AM.Free; // zbavime se docasnych
  GM.Free;
end;

//------------------------------------------------------------------------------

function RLog2(const X: TNumber; const Iters: Cardinal = 10): TTempNumber;
begin
  Result := RMul([RLn(X, Iters), MulInv(RLn2)]);
end;

//------------------------------------------------------------------------------

function RLog10(const X: TNumber; const Iters: Cardinal = 10): TTempNumber;
begin
  Result := RMul([RLn(X, Iters), MulInv(RLn10)]);
end;

//------------------------------------------------------------------------------

function RSin(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
var i: Integer;
    Current, Square: TNumber;
begin
  Current := TNumber.Create;
  Current.AssignValue(X);
  Square := TNumber.Create;
  Square.AssignValue(RMul([Current, MulInv(RPi)], 1, 2));
  Square.AssignValue(RMul([RRound(Square), RPi], -2)); // posuneme uhel do <-pi, pi>
  Current.AssignValue(RAdd([Current, Square]));
  Result := TTempNumber.Create(true);
  Result.AssignValue(Current);
  Square.AssignValue(RSqr(Current)); // ulozime druhou mocninu argumentu
  for i := 1 to Iters do begin
    CutTo(Result, 10); // zkratime vysledek 
    WriteLn(i);
    Current.AssignValue(RMul([Current, Square], -1, 2*i*(2*i+1))); // urcime dalsi clen rady
    CutTo(Current, 10); // ten zkratime
    AAdd(Result, Current); // a pricteme
  end;
  Current.Free;
  Square.Free;
end;

//------------------------------------------------------------------------------

function RCos(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
var i: Integer;
    Current, Square: TNumber;    // skoro stejne jako u RSin
begin
  Result := TTempNumber.Create(true);
  Result.AssignValue(1);
  Current := TNumber.Create;
  Current.AssignValue(X);
  Square := TNumber.Create;
  Square.AssignValue(RMul([Current, MulInv(RPi)], 1, 2));
  Square.AssignValue(RMul([RRound(Square), RPi], -2));
  Current.AssignValue(RAdd([Current, Square]));
  Square.AssignValue(RSqr(Current));
  Current.AssignValue(1);
  for i := 1 to Iters do begin
    CutTo(Result, 10);
    Current.AssignValue(RMul([Current, Square], -1, 2*i*(2*i-1)));
    CutTo(Current, 10);
    AAdd(Result, Current);
  end;
  Current.Free;
  Square.Free;
end;

//------------------------------------------------------------------------------

function RSec(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
begin
  Result := MulInv(RCos(X, Iters));
end;

//------------------------------------------------------------------------------

function RCsc(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
begin
  Result := MulInv(RSin(X, Iters));
end;

//------------------------------------------------------------------------------

function RTan(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
var i: Integer;
    SinX, CosX, Current, Pom: TNumber;
begin
  SinX := TNumber.Create;
  CosX := TNumber.Create;
  Current := TNumber.Create;
  Pom := TNumber.Create;
  Pom.AssignValue(X);
  Current.AssignValue(RMul([Pom, MulInv(RPi)], 1, 2)); // opet "kraceni" na zakl. uhel 
  Current.AssignValue(RMul([RRound(Current), RPi], -2));
  Pom.AssignValue(RAdd([Pom, Current]));
  Current.AssignValue(Pom);
  CosX.AssignValue(1);
  SinX.AssignValue(Pom);
  for i := 2 to Iters do begin
    Current.AssignValue(RMul([Current, Pom], 1, i)); // vypocitame dalsi clen rady
    CutTo(Current, 10); // zkratime ho
    if i and 1 = 0 then begin // rozhodneme, jestli patri do kosinu
      CutTo(CosX, 10);
      if (i shr 1) and 1 = 0 then begin // a jestli se pricte
        AAdd(CosX, Current);
      end else begin  // nebo odecte
        Current.Minus := not Current.Minus;
        AAdd(CosX, Current);
        Current.Minus := not Current.Minus;
      end;
    end else begin // nebo jestli patri do sinu
      CutTo(SinX, 10);
      if (i shr 1) and 1 = 0 then begin
        AAdd(SinX, Current);
      end else begin
        Current.Minus := not Current.Minus;
        AAdd(SinX, Current);
        Current.Minus := not Current.Minus;
      end;
    end;
  end;
  Result := RMul([SinX, MulInv(CosX)]); // vysledek je podilem
  SinX.Free;
  CosX.Free;   // uvolneni objektu
  Current.Free;
  Pom.Free;
end;

//------------------------------------------------------------------------------

function RCot(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
begin
  Result := MulInv(RTan(X, Iters));
end;

//------------------------------------------------------------------------------

function RSqrt(const X: TNumber; const TrySquare: Boolean; const Iters: Cardinal = 5): TTempNumber;
var V, Pom: TNumber;
    i: Integer;
    Temp, Square: TDynInt;
    IsSquare: Boolean;
begin
  if not X.Minus then begin
    if BIsZero(X.Citatel) then begin
      Result := TTempNumber.Create(true);
      if X is TTempNumber then X.Free;
    end else begin
      Pom := TNumber.Create;
      Pom.AssignValue(X);
      V := TNumber.Create;
      V.AssignValue(Pom);
      IsSquare := false;
      if TrySquare then begin
        DICreate(Temp);
        DICreate(Square);
        DICopy(Temp, V.Citatel);
        BShr(Temp, DynIntBitLength(Temp) div 2); // prvni odhad se najde bitovym posunem na pulku
        BMul(Square, Temp, Temp);
        if BGreater(Square, V.Citatel) then repeat  // test jestli je citatel ctverec
          BDec(Temp);           // jestli je odhad vetsi,
          BSub(Square, Temp);   // testujeme postupne cim dal mensi
          BSub(Square, Temp);   // druhe mocniny
          BDec(Square);
        until BGreaterEqual(V.Citatel, Square) else while BGreater(V.Citatel, Square) do begin
          BInc(Temp);          // v opacnem pripade
          BAdd(Square, Temp);  // testujeme vetsi
          BAdd(Square, Temp);
          BDec(Square);
        end;
        IsSquare := BEqual(Square, V.Citatel); // je druhou mocninou tehdy, pokud se rovna
        DICopy(V.Citatel, Temp);               // te, na ktere se to zastavilo
        DICopy(Temp, V.Jmenovatel);
        BShr(Temp, DynIntBitLength(Temp) div 2);
        BMul(Square, Temp, Temp);
        if BGreater(Square, V.Jmenovatel) then repeat  // to same provedeme se jmenovatelem
          BDec(Temp);
          BSub(Square, Temp);
          BSub(Square, Temp);
          BDec(Square);
        until BGreaterEqual(V.Jmenovatel, Square) else while BGreater(V.Jmenovatel, Square) do begin
          BInc(Temp);
          BAdd(Square, Temp);
          BAdd(Square, Temp);
          BDec(Square);
        end;
        IsSquare := IsSquare and BEqual(Square, V.Jmenovatel);
        DIFree(V.Jmenovatel);
        V.Jmenovatel := Temp;
      end else begin
        BShr(V.Citatel, DynIntBitLength(V.Citatel) div 2); 
        BShr(V.Jmenovatel, DynIntBitLength(V.Jmenovatel) div 2);
      end;
      Result := TTempNumber.Create(true);
      if IsSquare then Result.AssignValue(V) else begin
        for i := 1 to Iters do begin
          CutTo(V, 10); // zkracujeme V a pri tom provadime dalsi iterace Newtonovy metody 
          V.AssignValue(RMul([ RAdd([V, RMul([Pom, MulInv(V)])]) ], 1, 2)); 
        end;
        Result.AssignValue(V);
      end;
      V.Free;
    end;
  end else raise Exception.Create('Cannot compute square root of negative number');
end;

//------------------------------------------------------------------------------

function RPower(const Base, Power: TNumber; const Iters: Cardinal = 10): TTempNumber;
var Pom: TNumber;
begin
  if Base.Minus then begin  // pro zaporne zaklady
    if DIGet(Power.Jmenovatel, 0) and 1 = 1 then begin
      if (Power.Citatel.Length = 1) and (Power.Jmenovatel.Length = 1) // pro jednoduche exponenty
       and (DIGet(Power.Jmenovatel, 0) < 16) then begin            // pouzijeme mocninu a odmocninu
        Pom := TNumber.Create;
        if Power.Minus then Pom.AssignValue(RIntPower(Base, -DIGet(Power.Citatel, 0))) else
         Pom.AssignValue(RIntPower(Base, DIGet(Power.Citatel, 0)));
        Result := RIntRoot(Pom, DIGet(Power.Jmenovatel, 0));
        Pom.Free;
      end else begin
        if DIGet(Power.Citatel, 0) and 1 = 1 then     // jinak logaritmujeme
         Result := AddInv(RExp(RMul([Power, RLn(AddInv(Base), Iters)]))) else
         Result := RExp(RMul([Power, RLn(AddInv(Base), Iters)])); 
      end;
    end else raise Exception.Create('Cannot compute even root of negative number!');
  end else begin  // pro kladne zaklady
    if (Power.Citatel.Length = 1) and (Power.Jmenovatel.Length = 1)
     and (DIGet(Power.Jmenovatel, 0) < 16) then begin
      Pom := TNumber.Create;
      if Power.Minus then Pom.AssignValue(RIntPower(Base, -DIGet(Power.Citatel, 0))) else
       Pom.AssignValue(RIntPower(Base, DIGet(Power.Citatel, 0)));
      Result := RIntRoot(Pom, DIGet(Power.Jmenovatel, 0));
      Pom.Free;
    end else Result := RExp(RMul([Power, RLn(Base, Iters)]));
  end;
end;

//------------------------------------------------------------------------------

function RIntRoot(const X: TNumber; const Power: Integer; const Iters: Cardinal = 20): TTempNumber;
var i: Integer;
    V, Temp, Pom: TNumber;
    L: Cardinal;
begin
  if not (X.Minus and (Power and 1 = 0)) then begin // provedeme test na sudou odmocninu a zap. argument
    Pom := TNumber.Create;
    Pom.AssignValue(X);
    V := TNumber.Create;
    V.AssignValue(Pom);
    L := DynIntBitLength(V.Citatel);
    BShr(V.Citatel, Min(Round(L*(Power-1)/Power), L-1)); // urcime pocatecni odhad
    L := DynIntBitLength(V.Jmenovatel);                  // bitovym posunem
    BShr(V.Jmenovatel, Min(Round(L*(Power-1)/Power), L-1));
    Temp := TNumber.Create;
    for i := 1 to Iters do begin
      CutTo(V, 10); // probihaji iterace Newtonovy metody
      Temp.AssignValue(RMul([Pom, RIntPower(V, 1-Power)]));
      CutTo(Pom, 10);
      V.AssignValue(RMul([RAdd([ RMul(V, Power-1), Temp ])], 1, Power));
    end;
    Result := TTempNumber.Create(true);
    Result.AssignValue(V);
    V.Free;
    Pom.Free;
  end else raise Exception.Create('Cannot compute '+IntToStr(Power)+'th root of negative number');
end;

//------------------------------------------------------------------------------

function RArcSin(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
var i: Integer;
    Square, A, B, PrevA, PrevB, Prev2A, Prev2B, Pom: TNumber;
begin
  if BGreater(X.Citatel, X.Jmenovatel) then
   raise EInvalidArgument.Create('ArcSin argument out of <-1, 1>') else begin
    Pom := TNumber.Create;
    Pom.AssignValue(X);
    Square := TNumber.Create;
    A := TNumber.Create;
    B := TNumber.Create;
    PrevA := TNumber.Create;
    PrevB := TNumber.Create;
    Prev2A := TNumber.Create;
    Prev2B := TNumber.Create;
    Result := TTempNumber.Create(true);
    B.AssignValue(RSqr(Pom));
    A.AssignValue(RSqrt(RAdd([AddInv(B)], 1), false));
    Result.AssignValue(RMul([Pom, MulInv(RAdd(A, 1))], 2));
    Square.AssignValue(RMul([ B, MulInv(RAdd([ AddInv(B), RMul(A, 2) ], 2)) ]));
    PrevA.AssignValue(1);
    PrevB.AssignValue(1);
    A.AssignValue(RAdd(Square, 3));
    B.AssignValue(3);
    for i := 2 to Iters do begin
      CutTo(A, 10);
      CutTo(B, 10);
      Prev2A.AssignValue(PrevA);
      Prev2B.AssignValue(PrevB);
      PrevA.AssignValue(A);
      PrevB.AssignValue(B);
      A.AssignValue(RAdd([ RMul(PrevA, 2*i+1), RMul([Prev2A, Square], i*i) ]));
      B.AssignValue(RAdd([ RMul(PrevB, 2*i+1), RMul([Prev2B, Square], i*i) ]));
    end;
    AMul(Result, RMul([B, MulInv(A)]));
    A.Free;
    B.Free;
    Square.Free;
    PrevA.Free;
    PrevB.Free;
    Prev2A.Free;
    Prev2B.Free;
  end;
end;

//------------------------------------------------------------------------------

function RArcCos(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
begin
  Result := RAdd([AddInv(RArcSin(X, Iters)), RMul(RPi, 1, 2)]);
end;

//------------------------------------------------------------------------------

function RArcTan(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
var i: Integer;
    Square, A, B, PrevA, PrevB, Prev2A, Prev2B, bpart: TNumber;
begin
  if X.Minus then Result := AddInv(RArcTan(AddInv(X))) else begin
    if BGreaterEqual(X.Jmenovatel, X.Citatel) then begin
      Square := TNumber.Create;
      A := TNumber.Create;
      B := TNumber.Create;
      PrevA := TNumber.Create;
      PrevB := TNumber.Create;
      Prev2A := TNumber.Create;
      Prev2B := TNumber.Create;
      bpart := TNumber.Create;
      Result := TTempNumber.Create(true);
      Result.AssignValue(1);
      AMul(Result, X);
      Square.AssignValue(RSqr(X));
      PrevA.AssignValue(1);
      PrevB.AssignValue(1);
      A.AssignValue(RAdd(Square, 3));
      B.AssignValue(3);
      for i := 2 to Iters do begin
        CutTo(A, 10);
        CutTo(B, 10);
        Prev2A.AssignValue(PrevA);
        Prev2B.AssignValue(PrevB);
        PrevA.AssignValue(A);
        PrevB.AssignValue(B);
        bpart.AssignValue(RMul(Square, i*i));
        A.AssignValue(RAdd([ RMul(PrevA, 2*i+1), RMul([Prev2A, bpart]) ]));
        B.AssignValue(RAdd([ RMul(PrevB, 2*i+1), RMul([Prev2B, bpart]) ]));
      end;
      AMul(Result, RMul([B, MulInv(A)]));
      A.Free;
      B.Free;
      Square.Free;
      PrevA.Free;
      PrevB.Free;
      Prev2A.Free;
      Prev2B.Free;
    end else Result := RAdd([AddInv(RArcTan(MulInv(X))), RMul(RPi, 1, 2)]);
  end;
end;

//------------------------------------------------------------------------------

function RArcCot(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
begin
  Result := RArcTan(MulInv(X), Iters);
end;

//------------------------------------------------------------------------------

function RArcSec(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
begin
  if BGreaterEqual(X.Citatel, X.Jmenovatel) then
   Result := RArcCos(MulInv(X), Iters)
   else raise EInvalidArgument.Create('RArcSec argument in (-1, 1)');
end;

//------------------------------------------------------------------------------

function RArcCsc(const X: TNumber; const Iters: Cardinal = 20): TTempNumber;
begin
  if BGreaterEqual(X.Citatel, X.Jmenovatel) then
   Result := RArcSin(MulInv(X), Iters)
   else raise EInvalidArgument.Create('RArcCsc argument in (-1, 1)');
end;

//------------------------------------------------------------------------------

initialization

RLn2 := TNumber.Create; // nacitani RLn2
DISetLength(RLn2.Citatel, 40);
DISetLength(RLn2.Jmenovatel, 40);
Loader := TFileStream.Create('RLn2.dat', fmOpenRead);
for l := 0 to 39 do begin
  Loader.Read(B, 1);
  DISet(RLn2.Citatel, l, B);
end;
for l := 0 to 39 do begin
  Loader.Read(B, 1);
  DISet(RLn2.Jmenovatel, l, B);
end;
Loader.Free;

RLn10 := TNumber.Create; // nacitani RLn10
DISetLength(RLn10.Citatel, 40);
DISetLength(RLn10.Jmenovatel, 40);
Loader := TFileStream.Create('RLn10.dat', fmOpenRead);
for l := 0 to 39 do begin
  Loader.Read(B, 1);
  DISet(RLn10.Citatel, l, B);
end;
for l := 0 to 39 do begin
  Loader.Read(B, 1);
  DISet(RLn10.Jmenovatel, l, B);
end;
Loader.Free;

RPi := TNumber.Create; // nacitani RPi
DISetLength(RPi.Citatel, 40);
DISetLength(RPi.Jmenovatel, 40);
Loader := TFileStream.Create('RPi.dat', fmOpenRead);
for l := 0 to 39 do begin
  Loader.Read(B, 1);
  DISet(RPi.Citatel, l, B);
end;
for l := 0 to 39 do begin
  Loader.Read(B, 1);
  DISet(RPi.Jmenovatel, l, B);
end;
Loader.Free;


end.
