(*************************************************************************
 *  TranslationU.pas                                                     *
 *  Vladimr Slvik 2007                                                 *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Routines for i18n - loading text from gettext catalogs (without      *
 *    plurals), translating single strings. Also some helper routines.   *
 *                                                                       *
 *  -additional libraries: none                                          *
 *************************************************************************)
 
unit TranslationU;

interface
//==============================================================================

uses Classes;

type TTranslateFunction = function(const Generic: String): String;


function Translate(const Generic: String):String;

procedure TranslateStrings(const Target: TStrings);

procedure ChangeTranslation(const Stream: TStream); overload;

procedure ChangeTranslation(const FileName: String); overload;

procedure InvertTranslation;

function ChooseTranslationFile(const Language: String;
  ATranslations: TStrings; const ADefault: String): String;
// returns index of file that fits most for given language

var _: TTranslateFunction;

//==============================================================================
implementation

uses IniFiles, SysUtils, StrUtils, // inifiles for hashed stringlist
     StrU;

procedure AddTranslationItem(MsgId, MsgStr: String); forward;

function RemoveEscapes(const Escaped: String): String; forward;

var TranslationItems: TStrings;

(*
  How it works: TranslationItems is an internal hash of default+translated
  string pairs. It is filled after ChangeTranslation call, and the proper way of
  getting localized string is calling Translate which returns the string to be
  used. Another translation "task" are forms, this is done by TranslateComponent
  which (recursively) finds all components and translates their Hints and
  Captions. So, on the outside i18n looks like:

    first call ChangeTranslation
     -> some l10n is loaded
    call TranslateComponent on all forms
     -> the whole GUI is localized
     
    when "saying" something, pass result of Translate instead of default string
     -> all messages localized

  Unfortunately, there is no reasonable way to re-translate forms once they are
  localized, because the act of first localization changes all their properties
  and they cannot be *simply* re-hashed back or reloaded from their resources.
  The "simply" is keyword here - it would be well possible to write a parser for
  form resource data, but with the plan to move to Lazarus this simply won't
  do as it uses completely different approach to resources; rehashing forms back
  from the loaded translation before change also falls outside "easy", since
  such action requires some central coordination that processes all forms before
  the translation hash gets overwritten, but this is tricky as they may not
  exist at that time, leading to all possible kinds of problems. So, in short,
  this won't happen - only way to change language is to restart app.
*)
//------------------------------------------------------------------------------

function Translate(const Generic: String):String;
begin
  Result:= TranslationItems.Values[Generic];
  if Result = EmptyStr then Result:= Generic; 
end;

//------------------------------------------------------------------------------

procedure ChangeTranslation(const Stream: TStream); overload;
(* Complete parsing of gettext catalogs WITHOUT plurals or contexts. Based on
   old code of mine (PoU.pas), only this time without caring for comments and
   flags. *)
var i: Integer;
    S, M, MsgStr, MsgId: String;
    Lines: TStrings;
    CommentsLoaded: Boolean;
begin
  Lines:= TStringList.Create;
  Lines.LoadFromStream(Stream);
  CommentsLoaded:= False;
  with Lines do if Count > 0 then begin
    i:= 0;
    while i < Count do begin
      S:= Strings[i];
      if Length(S) > 0 then case S[1] of
        '#': begin
          if CommentsLoaded then begin
            MsgStr:= M;
            AddTranslationItem(MsgId, MsgStr);;
            CommentsLoaded:= False;
          end;
        end;
        'm', 'M': begin
          if UpperCase(Copy(S, 1, 7)) = 'MSGID "' then begin
            CommentsLoaded:= True;
            M:= Copy(S, 8, Length(S) - 8); // - 7 (msgid ") - 1 (")
          end;  
          if UpperCase(Copy(S, 1, 8)) = 'MSGSTR "' then begin
            MsgId:= M;
            M:= Copy(S, 9, Length(S) - 9); // - 8 (msgstr ") - 1 (")
          end;
        end;
        '"': begin
          M:= M + Copy(S, 2, Length(S) - 2); // - 1 (") - 1 (")
        end;
      end;
      Inc(i);
    end;
    if MsgId <> EmptyStr then begin
      MsgStr:= M;
      AddTranslationItem(MsgId, MsgStr);
    end;
  end;
  Lines.Free;
end;

//------------------------------------------------------------------------------

procedure ChangeTranslation(const FileName: String); overload;
var S: TStream;
begin
  try
    S:= TFileStream.Create(GoodSlashes(FileName), fmOpenRead);
    ChangeTranslation(S);
  finally
    S.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure AddTranslationItem(MsgId, MsgStr: String);
begin
  if (MsgId <> EmptyStr) and (MsgStr <> EmptyStr) and (MsgStr <> MsgId) then
    with TranslationItems do
      Values[RemoveEscapes(Utf8ToAnsi(MsgId))]:= RemoveEscapes(Utf8ToAnsi(MsgStr));
  // catalogs are (must be) in utf-8, convert!
end;

//------------------------------------------------------------------------------

function RemoveEscapes(const Escaped: String): String;
var p: Integer;
    S: String;
begin
  S:= Escaped;
  p:= PosEx('\', S, 1);
  while p <> 0 do begin
    Delete(S, p, 1); // remove the escape char \
    case S[p] of
      'n': S[p]:= #10; // \n
      '\': Inc(p); //     \\
      '"', '''': ; //     \" , \'
      't': S[p]:= ' '; // \t
    end;
    p:= PosEx('\', S, p);
  end;
  Result:= S;
end;

//------------------------------------------------------------------------------

function ChooseTranslationFile(const Language: String;
  ATranslations: TStrings; const ADefault: String): String;
var Translations: TStrings;
    S: String;
    i, Split, ToUse: Integer;
begin
  try
    Result:= ADefault;
    if ATranslations.Count < 1 then Abort; 
    Translations:= TStringList.Create;
    Translations.Assign(ATranslations);
    with ATranslations do for i:= 0 to Count - 1 do
      Strings[i]:= ExtractFileName(Strings[i]);
    StringsUpperCase(Translations);
    // prepare list of what we have to choose from
    S:= UpperCase(Language);
    ToUse:= Translations.IndexOf(S + '.PO');
    // try "as is"
    if ToUse > -1 then begin
      Result:= ATranslations[ToUse];
      // direct hit!
    end else begin
      // try substituting - with _
      S:= AnsiReplaceStr(S, '-', '_');
      ToUse:= Translations.IndexOf(S + '.PO');
      if ToUse > -1 then begin
        Result:= ATranslations[ToUse];
        // hit!
      end else begin
        // try substituting _ with -
        S:= AnsiReplaceStr(S, '_', '-');
        ToUse:= Translations.IndexOf(S + '.PO');
        if ToUse > -1 then begin
          Result:= ATranslations[ToUse];
          // hit!
        end else begin
          // and last try with basic language version
          ToUse:= Pos('_', S);
          if ToUse = 0 then Split:= Pos('-', S);
          // find where is _ or -
          if Split > 0 then begin
            // if some was there, lang string can be shortened just to basic version
            S:= Copy(S, 1, Split - 1);
            ToUse:= Translations.IndexOf(S + '.PO');
            if ToUse > -1 then begin
              Result:= ATranslations.Strings[ToUse];
            end;
          end;  
        end;
      end;
    end; 
  finally
    Translations.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure InvertTranslation;
var i: Integer;
    New: TStrings;
begin
  with TranslationItems do if Count > 0 then begin
    New:= THashedStringList.Create;
    for i:= 0 to Count - 1 do begin
      New.Values[ValueFromIndex[i]]:= Names[i];
    end;
    Free;
    TranslationItems:= New;
  end;
end;

//------------------------------------------------------------------------------

procedure TranslateStrings(const Target: TStrings);
var i: Integer;
begin
  with Target do if Count > 0 then for i:= 0 to Count - 1 do
    Strings[i]:= Translate(Strings[i]);
end;

//==============================================================================
initialization
  TranslationItems:= THashedStringList.Create;
  _:= Translate;

//==============================================================================
finalization
  TranslationItems.Free;

end.
