(*************************************************************************
 *  CalcUtilU.pas                                                        *
 *  Vladimr Slvik 2007-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Various small math (or pseudo-math :p) routines for Shades           *
 *                                                                       *
 *  -additional libraries: none                                          *
 *************************************************************************)

unit CalcUtilU;

{$INCLUDE ..\Switches.inc}
{t default -}

interface

uses Types;
//------------------------------------------------------------------------------

const ZeroRect: TRect = (Left: 0; Top: 0; Right: 0; Bottom: 0);
      // nothing
      OuterPoint: TPoint = (X: -1; Y: -1);

//------------------------------------------------------------------------------

function PtIs0(const APoint: TPoint): Boolean;
// Tests point for special constant OuterPoint.

function RIs0(const ARect: TRect): Boolean;
// Tests rectangle for special constant ZeroRect.

procedure CorrRect(var ARect: TRect);
// Flips the pairs of side coordinates to form a "correct" rectangle - more in
// procedure body, essentially so that Top > Bottom and Left > Right.

function RectWidth(const R: TRect): Integer;
function RectHeight(const R: TRect): Integer;

function AccumulateRect(const R1, R2: TRect): TRect;

function SubtractPoints(const P1, P2: TPoint): TPoint;
function AddPoints(const P1, P2: TPoint): TPoint;

function GrowRect(const R: TRect): TRect;
function ShrinkRect(const R: TRect): TRect;
// Grow or shrink by last row + column, very useful when juggling with rectangles
// between routines that do and don't consider these as part of it.   

function MakeRect1Valid(const R: TRect): TRect;
// make sure the rectangle is at least 1 valid row and column high and wide

function RectArea(const R: TRect): Integer; 

function DivPlus(const Dividend: Integer; const Divisor: Integer): Integer;
// Division "plus remainder" - if remainder is nonzero, add to division result

//==============================================================================
implementation

uses Math;
//------------------------------------------------------------------------------

function PtIs0(const APoint: TPoint): Boolean;
begin
  with APoint do Result:= (X = -1) and (Y = -1);
end;

//------------------------------------------------------------------------------

function RIs0(const ARect: TRect): Boolean;
// tests rectangle for "special constant"
begin
  with ARect do Result:= (Top = 0) and (Bottom = 0) and (Left = 0) and (Right = 0);
end;

//------------------------------------------------------------------------------

function DivPlus(const Dividend: Integer; const Divisor: Integer): Integer;
{$IFNDEF NO_ASSEMBLY}
// "Integer" expected signed !
asm
  mov ecx, edx // edx is target for division, value must be somewhere else
  xor edx, edx // erase
  idiv ecx // edx:eax / ecx = eax, edx
  cmp edx, 0
  je @end

  cmp eax, 0
  jb @neg

  inc eax
  ret

  @neg:
  dec eax

  @end:
end;
{$ELSE}
begin
  Result:= Dividend div Divisor;
  if Dividend mod Divisor > 1 then Inc(Result, Sign(Dividend * Divisor));
end;
{$ENDIF}

//------------------------------------------------------------------------------

procedure CorrRect(var ARect: TRect);
(* Flips the pairs of side coordinates if needed to make a correct rectangle =>
   Top < Bottom  AND  Left < Right

   +----------------------------+  |
   | TopLeft                    |  | Y numeric value
   |                            |  V
   |                BottomRight |
   +----------------------------+
     ----->
    X numeric value
*)
var Swap: Integer;
begin
  with ARect do begin
    if Top > Bottom then begin
      Swap:= Bottom;
      Bottom:= Top;
      Top:= Swap;
    end;
    if Left > Right then begin
      Swap:= Left;
      Left:= Right;
      Right:= Swap;
    end;
  end;
end;

//------------------------------------------------------------------------------

function RectHeight(const R: TRect): Integer;
begin
  with R do Result:= Bottom - Top;
end;

//------------------------------------------------------------------------------

function RectWidth(const R: TRect): Integer;
begin
  with R do Result:= Right - Left;
end;

//------------------------------------------------------------------------------

function AccumulateRect(const R1, R2: TRect): TRect;
var Switch: Integer;
begin
  Switch:= 0;
  if IsRectEmpty(R1) then Switch:= Switch or 1;
  if IsRectEmpty(R2) then Switch:= Switch or 2;
  case Switch of
    3: Result:= ZeroRect;
    2: Result:= R1;
    1: Result:= R2;
    0: UnionRect(Result, R1, R2);
  end;
end;

//------------------------------------------------------------------------------

function SubtractPoints(const P1, P2: TPoint): TPoint;
begin
  Result.X:= P1.X - P2.X;
  Result.Y:= P1.Y - P2.Y;
end;

//------------------------------------------------------------------------------

function AddPoints(const P1, P2: TPoint): TPoint;
begin
  Result.X:= P1.X + P2.X;
  Result.Y:= P1.Y + P2.Y;
end;

//------------------------------------------------------------------------------

function GrowRect(const R: TRect): TRect;
begin
  Result:= R;
  Inc(Result.Right);
  Inc(Result.Bottom);
end;

//------------------------------------------------------------------------------

function ShrinkRect(const R: TRect): TRect;
begin
  Result:= R;
  Dec(Result.Right);
  Dec(Result.Bottom);
end;

//------------------------------------------------------------------------------

function MakeRect1Valid(const R: TRect): TRect;
begin
  Result:= R;
  if Result.Top = Result.Bottom then Inc(Result.Bottom);
  if Result.Left = Result.Right then Inc(Result.Right);
end;

//------------------------------------------------------------------------------

function RectArea(const R: TRect): Integer;
// return area covered by this rectangle
begin
  Result:= (R.Top - R.Bottom - 1) * (R.Left - R.Right - 1);
end;

//------------------------------------------------------------------------------

end.
