(*************************************************************************
 *  ConfigU.pas                                                          *
 *  Vladimr Slvik 2010                                                 *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Fundamental config type definitions for Shades and routines to       *
 *    manipulate them.                                                   *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit ConfigU;

interface

{$INCLUDE ..\Switches.inc}
{t default -}

uses Classes, Controls, Forms, Graphics,
     IniFiles,
     GR32;

//------------------------------------------------------------------------------

const iftFile = 0;
      iftRegistry = 1;
      iftNone = 2;

type TIniFileKind = iftFile .. iftNone;

//------------------------------------------------------------------------------

type THighlightOptions = record
       BlendColor, DarkHlC, PlayerPriHlC, PlayerSecHlC: TColor32;
       Weight: Cardinal;
     end;

     TDarkOptions = record
       BlendColor: TColor32;
       Weight: Cardinal;
     end;

     TMouseOptions = record
       ScrollFactorX: Integer;
       ScrollFactorY: Integer;
       DragAreaCreate: Boolean;
       HorizontalScroll: Boolean;
       HorizontalPart: Integer;
       DiscardAreas: Boolean;
       AreaThreshold: Integer;
       CustomCursors: Boolean;
     end;

     TUndoOptions = record
       Restrict: Boolean;
       MaxLevel: Integer;
     end;

     TWindowOptions = record
       Top, Left, Width, Height: Integer;
       State: TWindowState;
     end;

     TInterfaceOptions = record
       ShowCloseBtn: Boolean;
       ToolBarAlign: TAlign;
       Sounds: Boolean;
       LangFile: String;
       DecimalIndicator: Boolean;
       ConfirmBkDelete: Boolean;
     end;

     TToolsOptions = record
       CustomRecolorer: Boolean;
     end;

     TSdiEdOptions = record
       HighlightMode: THighlightOptions;
       DarkMode: TDarkOptions;
       MouseOpt: TMouseOptions;
       DefZoomLevel: Single;
       PNGCompression: Integer;
       UndoOptions: TUndoOptions;
       InterfaceOptions: TInterfaceOptions;
       AddToRecent: Boolean;
       TimeStamp: Boolean;
       Window: TWindowOptions;
       Tools: TToolsOptions;
       IniFileKind: TIniFileKind;
       FirstRun: Boolean;
     end;
     PSdiEdOptions = ^TSdiEdOptions;

//------------------------------------------------------------------------------

procedure LoadOptions(const Ini: TCustomIniFile; var Options: TSdiEdOptions);

procedure LoadDefaultOptions(var Options: TSdiEdOptions);

procedure SaveOptions(const Ini: TCustomIniFile; var Options: TSdiEdOptions);

const UserDataDirPostfix = '/Shades/5.0/';

//------------------------------------------------------------------------------

procedure LoadBackgrounds(const FileName: String; TargetStrings: TStrings;
  IniKind: TIniFileKind);

procedure SaveBackgrounds(const FileName: String; SourceStrings: TStrings;
  IniKind: TIniFileKind);

//------------------------------------------------------------------------------

type TOptionsCompare = record
       HighlightChanged: Boolean;
       NightChanged: Boolean;
       MouseChanged: Boolean;
       UndoChanged: Boolean;
       ToolsChanged: Boolean;
     end;

function DiffOptions(const A, B: TSdiEdOptions): TOptionsCompare;

//==============================================================================
implementation

uses SysUtils, TypInfo, Registry,
     ColorU, StrU,
     SysLowU;
//==============================================================================

procedure LoadOptions(const Ini: TCustomIniFile; var Options: TSdiEdOptions);
var Section, S: String;
begin
  with INI, Options do begin
    Section:= 'HighlightMode';
    with HighlightMode do begin
      BlendColor:= HexToCl32(ReadString(Section, 'BlendColor', '00FFFFFF'));
      DarkHlC:= HexToCl32(ReadString(Section, 'NightPxColor', 'FFFF0000'));
      PlayerPriHlC:= HexToCl32(ReadString(Section, 'PlayerPriPxColor', 'FF0000FF'));
      PlayerSecHlC:= HexToCl32(ReadString(Section, 'PlayerSecPxColor', 'FF808000'));
      Weight:= ReadInteger(Section, 'Weight', 32);
    end;
    Section:= 'NightMode';
    with DarkMode do begin
      BlendColor:= HexToCl32(ReadString(Section, 'BlendColor', 'FF000000'));
      Weight:= ReadInteger(Section, 'Weight', 64);
    end;
    Section:= 'Mouse';
    with MouseOpt do begin
      ScrollFactorX:= ReadInteger(Section, 'ScrollFactorX', 50);
      ScrollFactorY:= ReadInteger(Section, 'ScrollFactorY', 50);
      DragAreaCreate:= ReadBool(Section, 'DragAreaCreate', False);
      HorizontalScroll:= ReadBool(Section, 'HorizontalScroll', False);
      HorizontalPart:= ReadInteger(Section, 'HorizontalPart', 25);
      DiscardAreas:= ReadBool(Section, 'DiscardAreas', False);
      AreaThreshold:= ReadInteger(Section, 'AreaThreshold', 60);
      CustomCursors:= ReadBool(Section, 'CustomCursors', True);
    end;
    Section:= 'Undo';
    with UndoOptions do begin
      Restrict:= ReadBool(Section, 'Restrict', False);
      MaxLevel:= ReadInteger(Section, 'MaxLevel', 10);
    end;
    Section:= 'Misc';
    DefZoomLevel:= ReadFloat(Section, 'ZoomLevel', 1);
    PNGCompression:= ReadInteger(Section, 'Compression', 9);
    AddToRecent:= ReadBool(Section, 'AddToRecent', True);
    TimeStamp:= ReadBool(Section, 'TimeStamp', True);
    FirstRun:= ReadBool(Section, 'FirstRun', True);
    Section:= 'Interface';
    with InterfaceOptions do begin
      ShowCloseBtn:= ReadBool(Section, 'CloseButton', False);
      S:= ReadString(Section, 'ToolBarAlign', 'Top');
      if UpperCase(S) = 'BOTTOM' then ToolBarAlign:= alBottom else ToolBarAlign:= alTop;
      Sounds:= ReadBool(Section, 'Sounds', True);
      LangFile:= ReadString(Section, 'LangFile', EmptyStr);
      DecimalIndicator:= ReadBool(Section, 'DecimalIndicator', False);
      ConfirmBkDelete:= ReadBool(Section, 'ConfirmBkDelete', True);
    end;
    Section:= 'Window';
    with Window do begin
      Top:= ReadInteger(Section, 'Top', 0);
      Left:= ReadInteger(Section, 'Left', 0);
      Width:= ReadInteger(Section, 'Width', 572);
      Height:= ReadInteger(Section, 'Height', 368);
      S:= ReadString(Section, 'State', 'wsNormal');
      State:= TWindowState(GetEnumValue(TypeInfo(TWindowState), S));
    end;
    Section:= 'Tools';
    with Tools do begin
      CustomRecolorer:= ReadBool(Section, 'CustomRecolorer', True);
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure LoadDefaultOptions(var Options: TSdiEdOptions);
begin
  with Options do begin
    with HighlightMode do begin
      BlendColor:= $00FFFFFF;
      DarkHlC:= $FFFF0000;
      PlayerPriHlC:= $FF0000FF;
      PlayerSecHlC:= $FF808000;
      Weight:= 32;
    end;
    with DarkMode do begin
      BlendColor:= $FF000000;
      Weight:= 64;
    end;
    with MouseOpt do begin
      ScrollFactorX:= 50;
      ScrollFactorY:= 50;
      DragAreaCreate:= False;
      HorizontalScroll:= True;
      HorizontalPart:= 25;
      DiscardAreas:= False;
      AreaThreshold:= 60;
      CustomCursors:= True;
    end;
    with UndoOptions do begin
      Restrict:= False;
      MaxLevel:= 0;
    end;
    DefZoomLevel:= 1;
    PNGCompression:= 9;
    AddToRecent:= True;
    TimeStamp:= True;
    FirstRun:= True;
    with InterfaceOptions do begin
      ShowCloseBtn:= False;
      ToolBarAlign:= alTop;
      Sounds:= True;
      LangFile:= EmptyStr;
      DecimalIndicator:= False;
      ConfirmBkDelete:= True;
    end;
    with Window do begin
      Top:= 0;
      Left:= 0;
      Width:= 572;
      Height:= 368;
      State:= wsNormal;
    end;
    with Tools do begin
      CustomRecolorer:= True;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure SaveOptions(const Ini: TCustomIniFile; var Options: TSdiEdOptions);
var Section: String;
begin
  with Ini, Options do begin
    Section:= 'HighlightMode';
    with HighlightMode do begin
      WriteString(Section, 'BlendColor', Cl32ToHex(BlendColor));
      WriteString(Section, 'NightPxColor', Cl32ToHex(DarkHlC));
      WriteString(Section, 'PlayerPriPxColor', Cl32ToHex(PlayerPriHlC));
      WriteString(Section, 'PlayerSecPxColor', Cl32ToHex(PlayerSecHlC));
      WriteInteger(Section, 'Weight', Weight);
    end;
    Section:= 'NightMode';
    with DarkMode do begin
      WriteString(Section, 'BlendColor', Cl32ToHex(BlendColor));
      WriteInteger(Section, 'Weight', Weight);
    end;
    Section:= 'Mouse';
    with MouseOpt do begin
      WriteBool(Section, 'DragAreaCreate', DragAreaCreate);
      WriteInteger(Section, 'ScrollFactorX', ScrollFactorX);
      WriteInteger(Section, 'ScrollFactorY', ScrollFactorY);
      WriteBool(Section, 'HorizontalScroll', HorizontalScroll);
      WriteInteger(Section, 'HorizontalPart', HorizontalPart);
      WriteBool(Section, 'DiscardAreas', DiscardAreas);
      WriteInteger(Section, 'AreaThreshold', AreaThreshold);
      WriteBool(Section, 'CustomCursors', CustomCursors);
    end;
    Section:= 'Undo';
    with UndoOptions do begin
      WriteBool(Section, 'Restrict', Restrict);
      WriteInteger(Section, 'MaxLevel', MaxLevel);
    end;
    Section:= 'Misc';
    WriteFloat(Section, 'Zoomlevel', DefZoomLevel);
    WriteInteger(Section, 'Compression', PNGCompression);
    WriteBool(Section, 'AddToRecent', AddToRecent);
    WriteBool(Section, 'TimeStamp', TimeStamp);
    WriteBool(Section, 'FirstRun', FirstRun);
    Section:= 'Interface';
    with InterfaceOptions do begin
      WriteBool(Section, 'CloseButton', ShowCloseBtn);
      case ToolBarAlign of
        alTop: WriteString(Section, 'ToolBarAlign', 'Top');
        else WriteString(Section, 'ToolBarAlign', 'Bottom');
      end;
      WriteBool(Section, 'Sounds', Sounds);
      WriteString(Section, 'LangFile', LangFile);
      WriteBool(Section, 'DecimalIndicator', DecimalIndicator);
      WriteBool(Section, 'ConfirmBkDelete', ConfirmBkDelete);
    end;
    Section:= 'Window';
    with Window do begin
      WriteInteger(Section, 'Top', Top);
      WriteInteger(Section, 'Left', Left);
      WriteInteger(Section, 'Width', Width);
      WriteInteger(Section, 'Height', Height);
      WriteString(Section, 'State',
        GetEnumName(TypeInfo(TWindowState), Integer(State)));
    end;
    Section:= 'Tools';
    with Tools do begin
      WriteBool(Section, 'CustomRecolorer', CustomRecolorer);
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure LoadBackgrounds(const FileName: String; TargetStrings: TStrings;
  IniKind: TIniFileKind);
var Ini: TCustomIniFile;
const Section = 'BackgroundPictures';
begin
  try
    case IniKind of
      iftFile: Ini:= TIniFile.Create(FileName);
      iftRegistry: Ini:= TRegistryIniFile.Create(FileName);
      iftNone: Ini:= nil;
    end;
    Assert(Assigned(Ini)); // !!!!
    Ini.ReadSectionValues(Section, TargetStrings);
    if IniKind = iftRegistry then
      CleanStrings(TargetStrings, '/', '\');
  finally
    Ini.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure SaveBackgrounds(const FileName: String; SourceStrings: TStrings;
  IniKind: TIniFileKind);
var Ini: TCustomIniFile;
    i: Integer;
const Section= 'BackgroundPictures';
begin
  try
    case IniKind of
      iftFile: Ini:= TIniFile.Create(FileName);
      iftRegistry: Ini:= TRegistryIniFile.Create(FileName);
      iftNone: Ini:= nil;
    end;
    Assert(Assigned(Ini)); // !!!!
    Ini.EraseSection(Section);
    if IniKind = iftRegistry then
      CleanStrings(SourceStrings, '\', '/');
    with SourceStrings do if Count > 0 then for i:= 0 to Count - 1 do
      Ini.WriteString(Section, Names[i], ValueFromIndex[i]);
    Ini.UpdateFile;
  finally
    Ini.Free;
  end;
end;

//------------------------------------------------------------------------------

function DiffOptions(const A, B: TSdiEdOptions): TOptionsCompare;
// Describes in helper type which parts changed; precise diffing can be left to
// caller since this is meant to reduce significantly clutter when determining
// eg. whether to repaint buffers.
begin
  Result.HighlightChanged:=
    (A.HighlightMode.BlendColor <> B.HighlightMode.BlendColor) or
    (A.HighlightMode.DarkHlC <> B.HighlightMode.DarkHlC) or
    (A.HighlightMode.PlayerPriHlC <> B.HighlightMode.PlayerPriHlC) or
    (A.HighlightMode.PlayerSecHlC <> B.HighlightMode.PlayerSecHlC) or
    (A.HighlightMode.Weight <> B.HighlightMode.Weight);
  Result.NightChanged:=
    (A.DarkMode.BlendColor <> B.DarkMode.BlendColor) or
    (A.DarkMode.Weight <> B.DarkMode.Weight);
  Result.MouseChanged:=
    (A.MouseOpt.ScrollFactorX <> B.MouseOpt.ScrollFactorX) or
    (A.MouseOpt.ScrollFactorY <> B.MouseOpt.ScrollFactorY) or
    (A.MouseOpt.DragAreaCreate <> B.MouseOpt.DragAreaCreate) or
    (A.MouseOpt.HorizontalScroll <> B.MouseOpt.HorizontalScroll) or
    (A.MouseOpt.HorizontalPart <> B.MouseOpt.HorizontalPart) or
    (A.MouseOpt.DiscardAreas <> B.MouseOpt.DiscardAreas) or
    (A.MouseOpt.AreaThreshold <> B.MouseOpt.AreaThreshold) or
    (A.MouseOpt.CustomCursors <> B.MouseOpt.CustomCursors);
  Result.UndoChanged:=
    (A.UndoOptions.Restrict <> B.UndoOptions.Restrict) or
    (A.UndoOptions.MaxLevel <> B.UndoOptions.MaxLevel);
  Result.ToolsChanged:=
    (A.Tools.CustomRecolorer <> B.Tools.CustomRecolorer);
end;

//------------------------------------------------------------------------------

end.
