(*************************************************************************
 *  RecolorU.pas                                                         *
 *  Vladimr Slvik 2010                                                 *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Machinery for recoloring.                                            *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit RecolorU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------

interface

//------------------------------------------------------------------------------

uses Classes, GR32;

//==============================================================================

type TTransformResult = record
       Value: TColor32;
       Valid: Boolean;
       Distance: Real; // in power!
     end;

type TMatchKernelBase = class
       function Transform(const AColor: TColor32): TTransformResult; virtual; abstract;
       constructor Create(const Data: TStrings); virtual; abstract;
     end;

//------------------------------------------------------------------------------

type TGrayKind = (gkNone, gkLuminosity, gkAverage);

type TGrayKernel = class(TMatchKernelBase)
       function Transform(const AColor: TColor32): TTransformResult; override;
       constructor Create(const Data: TStrings); override;
     private
       Kind: TGrayKind;
     end;

//------------------------------------------------------------------------------

type TBallKernel = class(TMatchKernelBase)
       function Transform(const AColor: TColor32): TTransformResult; override;
       constructor Create(const Data: TStrings); override;
     private
       Center: TColor32;
       Radius2: Real;
     end;

//------------------------------------------------------------------------------

type TBackgroundKernel = class(TMatchKernelBase)
       function Transform(const AColor: TColor32): TTransformResult; override;
       constructor Create(const Data: TStrings); override;
     private
       Tol2: Real;
     end;

//------------------------------------------------------------------------------

type TPointKernel = class(TMatchKernelBase)
       function Transform(const AColor: TColor32): TTransformResult; override;
       constructor Create(const Data: TStrings); override;
     private
       Color: TColor32;
     end;

//------------------------------------------------------------------------------

type TKeepKernel = class(TMatchKernelBase)
       function Transform(const AColor: TColor32): TTransformResult; override;
       constructor Create(const Data: TStrings); override;
     private
       Color: TColor32;
       Radius2: Real;
     end;

//==============================================================================

implementation

uses Math, SysUtils, Contnrs, StrU, ColorU, CoreTypeU, CoreLowU;

//==============================================================================

var ColorDistance2: TColor32Metric;

//------------------------------------------------------------------------------

function TGrayKernel.Transform;
begin
  Result.Valid:= True;
  Result.Value:= AColor;
  Result.Distance:= 0;
  case Kind of
    gkLuminosity: Result.Value:= GrayLum(AColor);
    gkAverage: Result.Value:= GrayAvg(AColor);
    else Result.Valid:= False;
  end;
  with Result do if Valid then Distance:= ColorDistance2(AColor, Value);
end;

//------------------------------------------------------------------------------

constructor TGrayKernel.Create;
begin
  Kind:= gkNone;
  if Pos('LUM', UpperCase(Data[1])) > 0 then begin
    Kind:= gkLuminosity;
  end else if Pos('AVG', UpperCase(Data[1])) > 0 then begin
    Kind:= gkAverage;
  end;
end;

//==============================================================================

function TBallKernel.Transform;
begin
  Result.Value:= AColor;
  Result.Distance:= 0;
  Result.Valid:= ColorDistance2(AColor, Center) <= Radius2;
end;

//------------------------------------------------------------------------------

constructor TBallKernel.Create;
begin
  Center:= Color32(HexToColor(Data[1]));
  Radius2:= Sqr(StrToFloat(Data[2]));
end;

//==============================================================================

function TBackgroundKernel.Transform;
begin
  Result.Value:= BackClrOpaque;
  Result.Distance:= ColorDistance2(BackClrOpaque, AColor);
  Result.Valid:= Result.Distance <= Tol2;
end;

//------------------------------------------------------------------------------

constructor TBackgroundKernel.Create;
begin
  Tol2:= Sqr(StrToFloat(Data[1]));
end;

//==============================================================================

function TPointKernel.Transform;
begin
  Result.Value:= Color;
  Result.Valid:= True;
  Result.Distance:= ColorDistance2(AColor, Color)
end;

//------------------------------------------------------------------------------

constructor TPointKernel.Create;
begin
  Color:= Color32(HexToColor(Data[1]));
end;

//==============================================================================

function TKeepKernel.Transform;
begin
  Result.Value:= Color;
  Result.Distance:= 0;
  Result.Valid:= ColorDistance2(AColor, Color) <= Radius2;
end;

//------------------------------------------------------------------------------

constructor TKeepKernel.Create;
begin
  Color:= Color32(HexToColor(Data[1]));
  Radius2:= Sqr(StrToFloat(Data[2]));
end;

//==============================================================================

initialization
  ColorDistance2:= Addr(ColorDistanceRGBEuclid2);
end.
