(*************************************************************************
 *  SysLowU.pas                                                          *
 *  Vladimr Slvik 2007-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Shades - routines for interaction with some low-level windows stuff  *
 *    and shell, mainly to separate the pointer operation and typecast   *
 *    mess and keep main code clean.                                     *
 *                                                                       *
 *  -additional libraries: none                                          *
 *************************************************************************)

unit SysLowU;

{$INCLUDE ..\Switches.inc}
{t default -}

interface

uses Classes, Graphics;

//------------------------------------------------------------------------------

function GetOSVersion: Integer;

function GetAppDataPath: String;
// returns path to "Application Data" folder - if possible

function GetTempPath: String;
// same for temp

procedure AddToRecent(const AFileName: String);
// adds file to recent documents

procedure WinOpen(const OwnerHandle: THandle; const What: String);
// opens the file or path ... or whatever, in this case URLs

function GetSystemLanguageName: String;
// returns language code of current locale

procedure GetFileIcon(FileName: String; Target: TIcon);
// extracts associated icon - meant to be used for executables

function DeleteFileSystemObject(const OwnerHandle: THandle;
  const FileName: String; const Confirm: Boolean): Boolean;

//------------------------------------------------------------------------------

type TStartThread = class(TThread)
     // for seamless usage of WinOpen - waiting for startup can be long
     public
       constructor CreateFast(const OwnerHandle: THandle;
         const ToStart, Parameters: String);
       constructor CreateWaiting(const OwnerHandle: THandle;
         const ToStart, Parameters: String; const AOnTerminate: TNotifyEvent);
     private
       FOwnerHandle: THandle;
       FToStart, FParams: String;
       FWait: Boolean;
     protected
       procedure Execute; override;
     end;

//------------------------------------------------------------------------------

var OSNumber: Integer;

const crCrossVert = 1;
      crCrossHorz = 2;
      crCrossDef = 3; // cross - general usage?
      crPenVert = 4;
      crPenHorz = 5;
      crPenDef = 6; // pen - line and pixel picking
      crMoveHorz = 7;
      crMoveVert = 8;
      crMoveDef = 9; // move - move
      {$MESSAGE WARN 'TODO: cursors for floodfills'}

procedure RegisterCursors;

//==============================================================================
implementation

uses Windows, ShellAPI, ShlObj, Sysutils, Forms,
     PngImage,
     ConstStrU, TranslationU;

var StrBuffer: array[1 .. MAX_PATH] of Char;
// API often need a buffer of varying size, where MAX_PATH is safe value, but
// overkill as well since it is sth. like 65536. Allocating 64kB of memory
// for local variables is not exactly great for performance, so use one global
// and hope threaded code won't need it as well.

//------------------------------------------------------------------------------

function GetOSVersion: Integer;
var OsVers: TOSVersionInfo;
begin
  // detect OS version, get rid of separated version numbering
  OsVers.dwOSVersionInfoSize:= SizeOf(TOSVersionInfo);
  if GetVersionEx(OsVers) then
    with OsVers do Result:= dwMajorVersion * 100 + dwMinorVersion
  else
    Result:= 0;
end;

//------------------------------------------------------------------------------

function GetAppDataPath: String;
var P: PChar;
begin
  P:= Addr(StrBuffer);
  if SHGetSpecialFolderPath(Application.Handle, P, CSIDL_APPDATA, False)
    then Result:= Trim(P) else
    raise Exception.Create(_('Could not retrieve CSIDL_APPDATA.')); //t+
end;

//------------------------------------------------------------------------------

function GetTempPath: String;
var P: PChar;
begin
  Result:= EmptyStr;
  P:= Addr(StrBuffer);
  Windows.GetTempPath(MAX_PATH, P); // needs unit specification!
  Result:= Trim(P);
end;

//------------------------------------------------------------------------------

procedure AddToRecent(const AFileName: String);
begin
  ShAddToRecentDocs(SHARD_PATH, PChar(AFileName));
end;

//------------------------------------------------------------------------------

constructor TStartThread.CreateFast(const OwnerHandle: THandle;
  const ToStart, Parameters: String);
begin
  inherited Create(True);
  FOwnerHandle:= OwnerHandle;
  FToStart:= ToStart;
  FParams:= Parameters;
  FreeOnTerminate:= True;
  OnTerminate:= nil;
  FWait:= False;
  Resume;
end;

//------------------------------------------------------------------------------

constructor TStartThread.CreateWaiting(const OwnerHandle: THandle;
  const ToStart, Parameters: String; const AOnTerminate: TNotifyEvent);
begin
  inherited Create(True);
  FOwnerHandle:= OwnerHandle;
  FToStart:= ToStart;
  FParams:= Parameters;
  FreeOnTerminate:= True;
  OnTerminate:= AOnTerminate;
  FWait:= True;
  Resume;
end;

//------------------------------------------------------------------------------

procedure TStartThread.Execute;
var ShInfo: PShellExecuteInfo;
begin
  New(ShInfo);
  with ShInfo^ do begin
    cbSize:= SizeOf(TShellExecuteInfo);
    fMask:= SEE_MASK_NOCLOSEPROCESS; // we DO want process handle
    Wnd:= FOwnerHandle;
    lpVerb:= nil; // defaults to OPEN
    lpFile:= PChar(FToStart);
    if FParams <> EmptyStr then
      lpParameters:= PChar(Fparams)
    else
      lpParameters:= nil;
    lpDirectory:= nil;
    nShow:= SW_SHOWNORMAL;
  end;
  ShellExecuteEx(ShInfo);
  if FWait then
    WaitForSingleObject(ShInfo.hProcess, INFINITE);
  Dispose(ShInfo);
end;

//------------------------------------------------------------------------------

procedure WinOpen(const OwnerHandle: THandle; const What: String);
begin
  TStartThread.CreateFast(OwnerHandle, What, EmptyStr);
end;

//------------------------------------------------------------------------------

procedure RegisterCursors;
// Custom cursors
begin
  with Screen do begin
    Cursors[crCrossVert]:= LoadCursor(hInstance, 'CRCROSSVERT');
    Cursors[crCrossHorz]:= LoadCursor(hInstance, 'CRCROSSHORZ');
    Cursors[crCrossDef]:= LoadCursor(hInstance, 'CRCROSSDEF');
    Cursors[crPenVert]:= LoadCursor(hInstance, 'CRPENVERT');
    Cursors[crPenHorz]:= LoadCursor(hInstance, 'CRPENHORZ');
    Cursors[crPenDef]:= LoadCursor(hInstance, 'CRPENDEF');
    Cursors[crMoveDef]:= LoadCursor(hInstance, 'CRMOVEDEF');
    Cursors[crMoveVert]:= LoadCursor(hInstance, 'CRMOVEVERT');
    Cursors[crMoveHorz]:= LoadCursor(hInstance, 'CRMOVEHORZ');
  end;
end;

//------------------------------------------------------------------------------
// based on code by Andreas Schmidt
// http://www.delphi3000.com/articles/article_2899.asp?SK=
// probably public domain

function GetSystemLanguageName: String;
var BytesNeeded: Integer;
    Locale: LCID;
begin
   Locale:= GetUserDefaultLCID;
   BytesNeeded:= GetLocaleInfo(Locale, LOCALE_SISO639LANGNAME, nil, 0);
   if BytesNeeded = 0 then RaiseLastOSError;
   Assert(BytesNeeded >= 1);
   SetLength(Result, BytesNeeded - 1); // we don't want to count the trailing #0
   GetLocaleInfo(Locale, LOCALE_SISO639LANGNAME, PChar(Result), Length(Result));
end;

//------------------------------------------------------------------------------
(* Extracts associated icon - meant to be used for executables. Thus, returning
   suitable something on error is OK.
*)

procedure GetFileIcon(FileName: String; Target: TIcon);
var P: PAnsiChar;
    W: Word;
    H: THandle;
begin
  P:= Addr(StrBuffer);
  StrLCopy(P, PChar(FileName), Length(FileName));
  W:= 0;
  H:= ExtractAssociatedIcon(Application.Handle, P, W);
  if H = 0 then raise EAbort.Create('Icon not found');
  Target.Handle:= H;
end;

//------------------------------------------------------------------------------
// code copied from an article by Zarko Gajic @ about.com : Delphi programming
// http://delphi.about.com/cs/adptips1999/a/bltip1199_2.htm

function DeleteFileSystemObject(const OwnerHandle: THandle;
  const FileName: String; const Confirm: Boolean): Boolean;
var
  SHFileOpStruct : TSHFileOpStruct;
  StrBuf: array[0..255] of Char;
begin
  try
    FillChar(SHFileOpStruct, SizeOf(SHFileOpStruct),0) ;
    FillChar(StrBuf, SizeOf(StrBuf), 0 ) ;
    StrPCopy(StrBuf, FileName) ;
    with SHFileOpStruct do begin
      Wnd:= OwnerHandle;
      pFrom:= @StrBuf;
      wFunc:= FO_DELETE;
      fFlags:= FOF_ALLOWUNDO or FOF_SILENT;
      if not Confirm then fFlags:= fFlags or FOF_NOCONFIRMATION;
    end;
    Result:= (SHFileOperation(SHFileOpStruct) = 0);
  except
    Result:= False;
  end;
end;

//==============================================================================

initialization
  OSNumber:= GetOSVersion;
  (*
     Possible OSNumber values - Source:
     http://msdn2.microsoft.com/en-us/library/ms724834.aspx
     retrieved 12/2006, updated 5/2009
     ----------------------------+---------------
     Operating system            | Version number
     ----------------------------+---------------
     Windows 7                   | 6.1
     Windows Server 2008 R2      | 6.1
     Windows Server 2008         | 6.0
     Windows Vista               | 6.0
     Windows Server 2003 R2      | 5.2
     Windows Server 2003         | 5.2
     Windows XP                  | 5.1
     Windows 2000                | 5.0
     Windows Me                  | 4.90
     Windows 98                  | 4.10
     Windows NT 4.0              | 4.0
     Windows 95                  | 4.0
     ----------------------------+---------------
  *)
end.
