(*************************************************************************
 *  ToolColorizeU.pas                                                    *
 *  Vladimr Slvik 2010                                                 *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Colorizing pencil tool - converts to special colours by painting     *
 *    over piels of arbitrary colour.                                    *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit ToolColorizeU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------

interface

uses Classes, ClassBaseU, GR32, Controls, ContNrs;

//------------------------------------------------------------------------------

type TColorizePencilTool = class(TMouseToolBase)
     private
       FKernels: TObjectList;
       procedure LoadConfig(const Target: TStrings);
     public
       constructor Create(AEngine: TPictureEngineBase); override;
       destructor Destroy; override;
       procedure MouseMove(const X, Y: Integer; const Shift: TShiftState);
         override;
       procedure MouseDown(const Button: TMouseButton; const X, Y: Integer;
         const Shift: TShiftState); override;
       procedure MouseUp(const Button: TMouseButton; const X, Y: Integer;
         const Shift: TShiftState); override;
       procedure MouseLeave; override;
       procedure Cancel; override;
     end;

//==============================================================================
implementation

uses Math, Windows, SysUtils, CoreLowU, CoreEngineU, SysLowU, ConfigU, CalcUtilU,
     RecolorU, StrU;
//==============================================================================

constructor TColorizePencilTool.Create(AEngine: TPictureEngineBase);
var Lines, Line: TStrings;
    i: Integer;
    Kernel: TMatchKernelBase;
begin
  inherited Create(AEngine);
  FKernels:= TObjectList.Create;
  try
    Lines:= TStringList.Create;
    Line:= TStringList.Create;
    LoadConfig(Lines);
    for i:= 0 to Lines.Count - 1 do begin
    SplitString(Lines[i], ' ', Line);
    if Line.Count > 1 then begin
      if Line[0] = 'background' then begin
        Kernel:= TBackgroundKernel.Create(Line);
        Assert(Assigned(Kernel));
        FKernels.Add(TObject(Kernel));
      end else if Line[0] = 'gray' then begin
        Kernel:= TGrayKernel.Create(Line);
        Assert(Assigned(Kernel));
        FKernels.Add(TObject(Kernel));
      end else if Line[0] = 'point' then begin
        Kernel:= TPointKernel.Create(Line);
        Assert(Assigned(Kernel));
        FKernels.Add(TObject(Kernel));
      end else if Line[0] = 'ball' then begin
        Kernel:= TBallKernel.Create(Line);
        Assert(Assigned(Kernel));
        FKernels.Add(TObject(Kernel));
      end else if Line[0] = 'keep' then begin
        Kernel:= TKeepKernel.Create(Line);
        Assert(Assigned(Kernel));
        FKernels.Add(TObject(Kernel));
      end;
    end;
  end;
  finally
    Line.Free;
    Lines.Free;
  end;
end;

//------------------------------------------------------------------------------

destructor TColorizePencilTool.Destroy;
begin
  FKernels.Free;
end;

//------------------------------------------------------------------------------

procedure TColorizePencilTool.MouseMove(const X, Y: Integer; const Shift: TShiftState);
const MaxResults = 48;
var R: TRect;
    C: TColor32;
    Results: array [0 .. MaxResults] of TTransformResult;
    BestIndex, i: Integer;
begin
  if FActive then try
    Assert(Assigned(FEngine));
    C:= FEngine.Picture.PixelS[X, Y];
    Results[MaxResults].Distance:= Infinity;

    BestIndex:= MaxResults;
    for i:= 0 to FKernels.Count - 1 do begin
      Assert(FKernels.Items[i] is TMatchKernelBase);
      Results[i]:= (FKernels.Items[i] as TMatchKernelBase).Transform(C);
      if Results[i].Valid and (Results[i].Distance < Results[BestIndex].Distance) then
        BestIndex:= i;
    end;
    C:= Results[BestIndex].Value;

    FEngine.Overlay.PixelS[X, Y]:= C;
    R:= Rect(X - 1, Y - 1, X + 2, Y + 2);
    FChangeArea:= AccumulateRect(FChangeArea, R);
    FEngine.UpdateRect(R);
    FEngine.RedrawRect(R);
  except
    // eat aborts
  end;
end;

//------------------------------------------------------------------------------

procedure TColorizePencilTool.MouseDown(const Button: TMouseButton;
    const X, Y: Integer; const Shift: TShiftState);
begin
  if (not FActive) and (Button = mbLeft) then begin // this one starts activity
    Assert(Assigned(FEngine));
    FActive:= True; // activate
    Self.MouseMove(X, Y, Shift); // and paint the first pixel
  end else if FActive and (Button = mbRight) then begin
    // stop activity
    Cancel;
  end;
end;

//------------------------------------------------------------------------------

procedure TColorizePencilTool.MouseUp(const Button: TMouseButton;
    const X, Y: Integer; const Shift: TShiftState);
begin
  if FActive and (Button = mbLeft) then begin
    FActive:= False;
    FEngine.CommitOverlay(FChangeArea);
    FEngine.RedrawRect(FChangeArea);
    FChangeArea:= ZeroRect;
  end;
end;

//------------------------------------------------------------------------------

procedure TColorizePencilTool.MouseLeave; {$I EMPTY.inc}
// do not react, keep painting

//------------------------------------------------------------------------------

procedure TColorizePencilTool.Cancel;
begin
  if FActive then begin
    FActive:= False;
    FEngine.DiscardOverlay;
    FEngine.UpdateRect(FChangeArea);
    FEngine.RedrawRect(FChangeArea);
    FChangeArea:= ZeroRect;
  end;
end;

//------------------------------------------------------------------------------

procedure TColorizePencilTool.LoadConfig(const Target: TStrings);
var Res: TStream;
    ConfFileName: String;
begin
  ConfFileName:= GetAppDataPath + UserDataDirPostfix + 'Recolor.cfg';
  if FEngine.Options.Tools.CustomRecolorer and (FEngine.Options.IniFileKind = iftFile) and FileExists(ConfFileName) then begin
    Target.LoadFromFile(ConfFileName);
  end else begin
    Res:= TResourceStream.Create(HInstance, 'RecolorConfig', RT_RCDATA);
    Target.LoadFromStream(Res);
  end;
end;

//------------------------------------------------------------------------------

end.
