unit LogU;

// Vladimr Slvik / Czech Republic / Brno
// ~ march 2005
// Delphi 7 Personal

// Object for easy logging into text file. If compared with simple Writeln
// calls, this proves more comfortable in ways of simplyfying some nonstandard
// tasks like indentation, writing arrays & lists and so on.

// BEWARE - if the destination file cannot be written to, you'll get some errors.

interface

uses Classes;

const DefLogLineLen = 5;       
      DefLogLineChr = '-';
      // default values for proc parameters

type TLogStyle = (        // good only for proc. Log(args: array of string ... );
       lsLines,           // - each Args[i] is a new line
       lsSpacedWords,     // - Args[i] + #32 + Args[i+1] + ...
       lsCommaSeparated,  // - Args[i] + ',' + Args[i+1] + ...
       lsNumbered,        // - Counter:3 , Args[i] & newline
       lsNoSpace          // - direct concatenation
     );

type
 TLog = class
         constructor Create(const FileName: String);
         procedure Log(const Args: array of String;
           const Style: TLogStyle = lsLines); overload;
         procedure Log(const Arg: TStrings); overload;
         procedure Log(const Arg: String); overload;
         procedure Log; overload;
         procedure LogNoIndent(const Arg: String);
         procedure LogLine(const C: Char; const L: Byte = DefLogLineLen); overload;
         procedure LogLine(const L: Byte = DefLogLineLen); overload;
         procedure IncIndentLevel;
         procedure DecIndentLevel;
         destructor Destroy; override;
        private
         FFile: TextFile;
         FFileName, FIndentStr: String;
         FEnabled: Boolean;
         FIndent, FLastIndent: Byte;
         FCommaSeparatorChar: Char;
         FIndentMulti: Byte;
         procedure FsetIndent;
         procedure FsetIndMulti(const M: Byte);
         function FgetRealIndLen: Integer;
        public
         property FileName: String read FFileName;
         property IndentLevel: Byte read FIndent write FIndent;
         property Enabled: Boolean read FEnabled write FEnabled;
         property CommaSeparatorChar: Char read FCommaSeparatorChar
           write FCommaSeparatorChar;
         property IndentMultiplier: Byte read FIndentMulti write FsetIndMulti;
         property RealIndentStringLenght: Integer read FgetRealIndLen;
        end;

(*
  TLog class
  ----------
  * Create, Free, Destroy - clear ;)
  * FileName - read only, run-time access to name of file being used for logging.
  * Indent - how many spaces are inserted in the beginning of each line. A bit
    useless, but still you might want to set it "hard way" s it's there.
  * IncIndent, DecIndent - use these to change indent in a relative way. In the
    first app which used TLog, every procedure or function began with logging
    routine's name. The whole content is between IncIndent and DecIndent so that
    the logged structure shows the nesting of calls etc. looking like :

    MainFrm.ShapeMouseDown(Shape35)
    MainFrm.SetPriColor($000000)
     MainFrm.SetHexInfo / Hex: $000000 / $808080 / $FFFFFF
     MainFrm.tcNearColorsChange
      MainFrm.ReloadNearColors($000000)
    MainFrm.ShapeMouseDown(Shape01)
    MainFrm.twCloseQuery(twGeneral)

    ... got it? It is more useful to do it this way!

  * Enabled - well, it is the same "enabled" as of any VCL control.
  * CommaSeparatorChar - char used when separating strings in proc. Log
  * Log - procedure which "does it". More overloaded versions, explained in
    implementations.

*)

//==============================================================================
implementation
uses SysUtils;
//==============================================================================

constructor TLog.Create(const FileName: String);
begin
  inherited Create;
  FFileName:= FileName;
  AssignFile(FFile, FFileName);
  Rewrite(FFile);
  FIndentStr:= EmptyStr;
  FIndent:= 0;
  FIndentMulti:= 1;
  FEnabled:= True;
end;

//------------------------------------------------------------------------------

destructor TLog.Destroy;
begin
  try
    CloseFile(FFile);
  finally
    inherited Destroy;
  end;
end;

//------------------------------------------------------------------------------

procedure TLog.FsetIndMulti(const M: Byte);
begin
  FLastIndent:= 0; // force recalculation of string
  FIndentMulti:= M;
  Writeln(FFile, '> Log: changed indent multiplier to ', M);
end;

//------------------------------------------------------------------------------

procedure TLog.FSetIndent;
begin
  if FIndent <> FLastIndent then begin 
    if FIndent = 0 then FIndentStr:= EmptyStr else
      FIndentStr:= StringOfChar(#32, FIndent * FIndentMulti);
    // set the desired lenght
    FLastIndent:= FIndent; // update lenght info
   end;
end;

//------------------------------------------------------------------------------

procedure TLog.Log(const Args: array of String; const Style: TLogStyle = lsLines);
// The most complicated version, takes open array and formats it according to
// Style param. Please note that I did not include indentation in examples so as
// not to make it overcomplicated.
var i, max, counter: Integer;
begin
  if FEnabled then begin
    FSetIndent;
    max:= High(Args);
    // ^ Compute the upper bound; needed in variable so that I can check whether
    // I'm at the end.
    case Style of
      lsLines: begin
        // -> every array part should be treated as single log entry (one line)
        // example: ['a','b','c'] results in 'aCRLFbCRLFcCRLF' where CRLF is new
        // line #10, #13
        for i:= Low(Args) to max do Writeln(FFile, FIndentStr, Args[i]);
      end;
      lsSpacedWords: begin
        // -> array parts should be on one line as separate words
        // example: ['a','b','c'] results in 'a b c'
        Write(FFile, FIndentStr); // start the line with indent
        for i:= Low(Args) to max do if not (i = max) then
          Write(FFile, Args[i], ' ') else Writeln(FFile, Args[i]);
      end;
      lsCommaSeparated: begin
        // -> same as above, but instead of space is used some char,
        // ',' (comma) as default value.
        // example: ['a','b','c'] might result in 'a,b,c'
        Write(FFile, FIndentStr);
        for i:= Low(Args) to max do if not (i = max) then
          Write(FFile, Args[i], FCommaSeparatorChar) else Writeln(FFile, Args[i]);
      end;
      lsNumbered: begin
        // -> "numbered list" of array parts
        // example: ['a','b','c'] results in '  1 aCRLF  2 bCRLF  3 cCRLF'
        counter:= 0; // reset
        for i:= Low(Args) to max do begin
          Writeln(FFile, FindentStr, counter:3, ' ', Args[i]); // write item
          inc(Counter);
        end;
      end;
      lsNoSpace: begin
        Write(FFile, FIndentStr);
        for i:= Low(Args) to max - 1 do Write(FFile, Args[i]);
        Writeln(FFile, Args[max]);
      end;
    end;
  end; 
end;

//------------------------------------------------------------------------------

procedure TLog.Log(const Arg: TStrings);
// log more than just one string, basically the same as ver. below
var i: integer;
begin
  if FEnabled and (Arg.Count > 0) then begin
    FsetIndent;
    for i:= 0 to Arg.Count - 1 do Writeln(FFile, FIndentStr, Arg.Strings[i]);
  end;
end;

//------------------------------------------------------------------------------

procedure TLog.Log(const Arg:String);
// single string version, no comment
begin
  if FEnabled then begin
    FSetIndent;
    Writeln(FFile, FIndentStr, Arg);
  end; 
end;

//------------------------------------------------------------------------------

procedure TLog.Log;
// the most simple one - just empty line
begin
  if FEnabled then Writeln(FFile);
end;

//------------------------------------------------------------------------------

procedure TLog.LogNoIndent(const Arg: String);
begin
  if FEnabled then Writeln(FFile, Arg);
end;

//------------------------------------------------------------------------------

procedure TLog.LogLine(const C: Char; const L: Byte);
// logs a "line" using supplied char and lenght (or default) - see constants
var S: String;
begin
  if FEnabled then begin
    FSetIndent;
    S:= StringOfChar(C, L);
    Writeln(FFile, FIndentStr, S); // write it there and done
  end;
end;

//------------------------------------------------------------------------------

procedure TLog.LogLine(const L: Byte);
// logs a "line" using supplied lenght (or default) and default char - see constants
var S: String;
begin
  if FEnabled then begin
    FsetIndent;
    S:= StringOfChar(DefLogLineChr, L);
    Writeln(FFile, FIndentStr, S); // write it there and done
  end;
end;

//------------------------------------------------------------------------------

procedure TLog.IncIndentLevel;
begin
  if FIndent < 255 then inc(FIndent);
end;

//------------------------------------------------------------------------------

procedure TLog.DecIndentLevel;
begin
  if FIndent > 0 then dec(FIndent);
end;

//------------------------------------------------------------------------------

function TLog.FgetRealIndLen: Integer;
begin
  Result:= Length(FIndentStr);
end;

//------------------------------------------------------------------------------

end.
