unit StrU;
// some useful string routines
//==============================================================================
interface
//==============================================================================

(*
{$IFDEF STRU_USE_TNT_UNICODE}
{$ELSE}
{$ENDIF}
dummy lines for copy-paste, ignore
*)


{-$DEFINE STRU_USE_TNT_UNICODE}
(*
  Add or remove '-' in front of '$' to disable/enable Tnt Unicode string support
  - like these:
  
  {$DEFINE STRU_USE_TNT_UNICODE} // ON (default state)
  {-$DEFINE STRU_USE_TNT_UNICODE} // OFF

  If switched on, this unit needs the following free additional libraries:

  Tnt Delphi Unicode Controls
  Copyright (c) 2002-2005, Troy Wolbrink (troy.wolbrink@tntware.com)
  http://www.tntware.com/delphicontrols/unicode/   
*)

{$IFDEF STRU_USE_TNT_UNICODE}
uses Classes, SysUtils, TntWideStrings; // if this causes errors, look up ^
{$ELSE}
uses Classes, SysUtils;
{$ENDIF}

type TSetOfChar = set of AnsiChar;

const CommonBadChars: TSetOfChar = [#10, #13, #7];
      PathBadChars: TSetOfChar = [':', '*', '?', '"', '|', '<', '>'];
      FileNameBadChars: TSetOfChar = ['/', '\', ':', '*', '?', '"', '|', '<', '>'];

type TJoinStringCallbackFunction = function (const First, Second: String): String;

const scCaseSensitive = 0; // no conversion performed
      scCaseInsensitive = 1; // both uppercased
      scConvertBaseUp = 2; // the rest is self-explanatory
      scConvertSearchedUp = 3;
      scConvertBaseDown = 4;
      scConvertSearchedDown = 5;
      

type TStringComparison = scCaseSensitive .. scConvertSearchedDown;

type TCharCost = function (const First, Second: Char): Double; 
                                  
//------------------------------------------------------------------------------
      
procedure SplitString(const strInput: String; const Separator: Char;
  const Strings: TStrings); overload;

procedure SplitString(const strInput, Separator: String;
  const Strings: TStrings); overload;

function CleanString(const S:String; const Bad: TSetOfChar;
  const Good: Char): String; overload;

function CleanString(const S:String; const Bad: Char; const Good: Char): String;
  overload;

function StringsHaveAllItems(const Base, SearchedParts: TStrings;
  const Comparison: TStringComparison = scCaseSensitive): Boolean;

procedure MergeStrings(Target, Update: TStrings);

function WinBkSlashes(const Path: String): String;

procedure DeleteEmptyStrings(const Target: TStrings);

function JoinStrings(const Source: TStrings; const Separator: String): String;
  overload;

function JoinStrings(const Source: TStrings;
  CallBack: TJoinStringCallbackFunction): String; overload;

{$IFDEF STRU_USE_TNT_UNICODE}
function JoinStrings(const Source: TWideStrings;
  const Separator: WideString): WideString; overload;
{$ENDIF}

function LevenshteinDistance(const A, B: String): Integer; overload;

function LevenshteinDistance(const A, B: String;
  GapCost, DiffCost: TCharCost): Double; overload;


//==============================================================================
implementation

uses StrUtils, Math;
//==============================================================================

function CleanString(const S:String; const Bad: TSetOfChar;
  const Good: Char): String; overload;
var i, L: LongInt;
begin
  L:= Length(S);
  Result:= S;
  if L > 1 then for i:= 1 to L do if Result[i] in Bad then Result[i]:= Good;
end;

//------------------------------------------------------------------------------

function CleanString(const S: String; const Bad: Char; const Good: Char): String; overload;
var i, L: LongInt;
begin
  L:= Length(S);
  Result:= S;
  if L > 1 then for i:= 1 to L do if Result[i] = Bad then Result[i]:= Good;
end;
                                 
//------------------------------------------------------------------------------

procedure SplitString(const strInput: String; const Separator: Char;
  const Strings: TStrings);
var i: Integer;
    c: Char;
    S: String;
begin
  S:= EmptyStr;
  Strings.Clear;
  for i:= 1 to Length(strInput) do begin
    c:= strInput[i];
    if c = Separator then begin
      Strings.Append(S);
      S:= EmptyStr;
    end else S:= S + c;
  end;
  if S <> EmptyStr then Strings.Add(S);
end;

//------------------------------------------------------------------------------

procedure SplitString(const strInput, Separator: String;
  const Strings: TStrings); overload;
var Head, sepLen, curStart: Integer;
    FirstPass: Boolean;
begin
  Strings.Clear;
  sepLen:= Length(Separator);
  Head:= 1;
  FirstPass:= True;
  while Head <> 0 do begin
    if FirstPass then begin
      curStart:= Head;
      FirstPass:= False;
    end else begin
      curStart:= Head + sepLen;
      Inc(Head);
    end;  
    Head:= PosEx(Separator, strInput, Head);
    if head <> 0 then Strings.Append(Copy(strInput, curStart, Head - curStart))
    else Strings.Append(Copy(strInput, curStart, Length(strInput) - curStart + 1));
  end;
end;

//------------------------------------------------------------------------------

function StringsHaveAllItems(const Base, SearchedParts: TStrings;
  const Comparison: TStringComparison): Boolean;
(*
   If "Base" contains all items from "SearchedParts" then result is positive
   (true); Comparison determines case sensitivity.
*)
var i, j: integer;
    ThisItemHere: boolean;
begin
  Result:= False;
  if (Base.Count > 0) and (SearchedParts.Count > 0) then begin
    Result:= True;
    for i:= 0 to SearchedParts.Count - 1 do begin
      ThisItemHere:= False;
      case Comparison of
        scCaseSensitive: ThisItemHere:= Base.IndexOf(SearchedParts.Strings[i]) > -1;
        scCaseInsensitive: begin
          for j:= 0 to Base.Count - 1 do if
              UpperCase(Base.Strings[j]) = UpperCase(SearchedParts.Strings[i])
              then begin
            ThisItemHere:= True;
            Break; // Skip cycle searching inside Base - not the outer, in
            // SearchedParts! 
          end;
        end;
        scConvertBaseUp: begin
          for j:= 0 to Base.Count - 1 do if
              UpperCase(Base.Strings[j]) = SearchedParts.Strings[i]
              then begin
            ThisItemHere:= True;
            Break;
          end;
        end;
        scConvertSearchedUp: begin
          for j:= 0 to Base.Count - 1 do if
              Base.Strings[j] = UpperCase(SearchedParts.Strings[i])
              then begin
            ThisItemHere:= True;
            Break;
          end;
        end;
        scConvertBaseDown: begin
          for j:= 0 to Base.Count - 1 do if
              LowerCase(Base.Strings[j]) = SearchedParts.Strings[i]
              then begin
            ThisItemHere:= True;
            Break;
          end;
        end;
        scConvertSearchedDown: begin
          for j:= 0 to Base.Count - 1 do if
              Base.Strings[j] = LowerCase(SearchedParts.Strings[i])
              then begin
            ThisItemHere:= True;
            Break; 
          end;
        end;
      end; // case
      if not ThisItemHere then begin
        Result:= False;
        Exit;
      end;
    end; // for SearchedParts
  end; // if count > 0
end;

//------------------------------------------------------------------------------

procedure MergeStrings(Target, Update: TStrings);
var i: Integer;
    S: String;
begin
  // if target is empty, just add all; if not, work
  if Target.Count > 0 then with Update do if Count > 0 then for i:= 0 to Count - 1 do begin
    S:= Strings[i];
    if Target.IndexOf(S) = -1 then Target.AddObject(S, Objects[i]);
  end else Target.AddStrings(Update);
end;

//------------------------------------------------------------------------------

function JoinStrings(const Source: TStrings; const Separator: String): String;
var S: String;
    i: Integer;
begin
  with Source do if Count > 0 then begin
    if Count > 1 then begin
      S:= EmptyStr;
      for i:= 0 to Count - 1 do if i = 0 then S:= Strings[0]
        else S:= S + Separator + Strings[i];
      Result:= S; // >= 2
    end else Result:= Strings[0]; // = 1
  end else Result:= EmptyStr; // = 0
end;

//------------------------------------------------------------------------------

function JoinStrings(const Source: TStrings;
  CallBack: TJoinStringCallbackFunction): String;
// CallBack should return the proper separator string for supplied strings
var S: String;
    i: Integer;
begin
  with Source do if Count > 0 then begin
    if Count > 1 then begin
      S:= EmptyStr;
      for i:= 0 to Count - 1 do if i = 0 then S:= Strings[0]
        else S:= S + CallBack(Strings[i - 1], Strings[i]) + Strings[i];
      Result:= S; // >= 2
    end else Result:= Strings[0]; // = 1
  end else Result:= EmptyStr; // = 0
end;

//------------------------------------------------------------------------------

{$IFDEF STRU_USE_TNT_UNICODE}
function JoinStrings(const Source: TWideStrings;
  const Separator: WideString): WideString;
var S: WideString;
    i: Integer;
begin
  with Source do if Count > 0 then begin
    if Count > 1 then begin
      S:= EmptyWideStr;
      for i:= 0 to Count - 1 do if i = 0 then S:= Strings[0]
        else S:= S + Separator + Strings[i];
      Result:= S; // >= 2
    end else Result:= Strings[0]; // = 1
  end else Result:= EmptyStr; // = 0
end;
{$ENDIF}

//------------------------------------------------------------------------------

function WinBkSlashes(const Path: String): String;
begin
  Result:= CleanString(Path, '/', '\');
end;

//------------------------------------------------------------------------------

function LevenshteinDistance(const A, B: String): Integer;
var A_len, B_len: Integer;
    Buf: array of array of Integer;
    i, j: Integer;
    Cost: Integer;
begin           
    A_len:= Length(A);
    B_len:= Length(B);
    // startup calculations
    if A_len = 0 then begin
      Result:= B_len;
      Exit;
    end;
    if B_len = 0 then begin
      Result:= A_len;
      Exit;
    end;
    // test for invalid lenghts
    SetLength(Buf, B_len + 1);
    for i:= 0 to B_len do SetLength(Buf[i], A_len + 1);
    // create matrix; indexing: [row, column]
    for i:= 0 to A_len do Buf[0, i]:= i;
    for i:= 0 to B_len do Buf[i, 0]:= i;
    // initialize zero row and column
    for i:= 1 to A_len do for j:= 1 to B_len do begin
      if A[i] = B[j] then Cost:= 0 else Cost:= 1;
      Buf[j, i]:= MinIntValue([
        Buf[j, i - 1] + 1,
        Buf[j - 1, i] + 1,
        Buf[j - 1, i - 1] + Cost
      ]);
    end;
    // fill matrix  
    Result:= Buf[B_len, A_len]; 
end;

//------------------------------------------------------------------------------

function LevenshteinDistance(const A, B: String;
  GapCost, DiffCost: TCharCost): Double;
var A_len, B_len: Integer;
    Buf: array of array of Double;
    i, j: Integer;
    Gap: Double; 
begin           
    A_len:= Length(A);
    B_len:= Length(B);
    // startup calculations
    if A_len = 0 then begin
      Result:= B_len;
      Exit;
    end;
    if B_len = 0 then begin
      Result:= A_len;
      Exit;
    end;
    // test for invalid lenghts
    SetLength(Buf, B_len + 1);
    for i:= 0 to B_len do SetLength(Buf[i], A_len + 1);
    // create matrix; indexing: [row, column]
    for i:= 0 to A_len do Buf[0, i]:= i;
    for i:= 0 to B_len do Buf[i, 0]:= i;
    // initialize zero row and column
    for i:= 1 to A_len do for j:= 1 to B_len do begin
      Gap:= GapCost(A[i], B[j]); // user made, call it only once
      Buf[j, i]:= MinValue([
        Buf[j, i - 1] + Gap,
        Buf[j - 1, i] + Gap,
        Buf[j - 1, i - 1] + DiffCost(A[i], B[j])
      ]);
    end;
    // fill matrix  
    Result:= Buf[B_len, A_len]; 
end;

//------------------------------------------------------------------------------

procedure DeleteEmptyStrings(const Target: TStrings);
var i: Integer;
begin
  i:= 0;
  with Target do if Count > 0 then while i < Count do
    if Strings[i] = EmptyStr then Delete(i) else Inc(i);
end;

//------------------------------------------------------------------------------

end.
