program Tbz;

{$Apptype Console}

{$R 'help.res' 'help.rc'}
{%File 'help.txt'}
{%File 'libbz2.dll'}

{-$DEFINE DEBUG} // debug build with logging

{$IFDEF DEBUG}
  {$DEFINE CLOSEWAIT} // don't close console
{$ENDIF}


uses
  SysUtils,
  Classes,
  LibTar,
  Bzip2,
  ShellAPI,
  StrU,
  {$IFDEF DEBUG}
  LogU,
  {$ENDIF}
  ConsoleU;

var Tar: TTarWriter;
    Bzip: TBZip2;

    Dummy: TFileStream;
    OutFile: TFileStream;

    i, RootPathLen, UsedBlockSize, UsedBufferSize: Integer;

    tempPath, tempFName, OpSpec, InputSpec, OutputSpec, WorkPath, S: String;

    FilesSize, DriveFree: Int64;
    GoCompress: Boolean;

    Files, Path: TStringList;
    
    (*tempFNfmt: TFormatSettings;*)

    x, y, temp_y : Word;

    {$IFDEF DEBUG}
    Log: TLog;
    {$ENDIF}

const FileAttrs = faReadOnly or faArchive or faSysFile or faHidden;
      FolderAttrs = FileAttrs or faDirectory;
      DefaultBlockSize = 5;
      DefaultBufferSize = 32;

//------------------------------------------------------------------------------

function SizeOfFile(const FileName: String): Int64;
var F: TFileStream;
begin
  {$IFDEF DEBUG}
    Log.Log(['SizeOfFile("', FileName, '");'], lsNoSpace);
    Log.IncIndentLevel;
  {$ENDIF}
  try
    try
      F:= TFileStream.Create(WinBkSlashes(FileName), fmOpenRead);
      Result:= F.Size;
    except on E: Exception do
      Result:= High(Int64); // give too big if inaccessible
    end;
  finally
    F.Free;
  end;
  {$IFDEF DEBUG}
    Log.Log(['Result = ', IntToStr(Result), ';'], lsNoSpace);
    Log.DecIndentLevel;
  {$ENDIF}
end;        

//------------------------------------------------------------------------------

procedure HelpInfo;
var i: Integer;
    RS: TResourceStream;
begin
  RS:= TResourceStream.Create(hInstance, 'help', 'RT_TEXT');
  with Files do begin
    Clear;
    LoadFromStream(RS);
    for i:= 0 to Files.Count - 1 do
      Writeln(Files.Strings[i]);
    Free;
  end;
  RS.Free;
  {$IFDEF CLOSEWAIT}
    Readln;
  {$ENDIF}
  Halt(1);
end;

//------------------------------------------------------------------------------

function GetCurPath: String;
var i: Integer;
begin
  Result:= EmptyStr;
  with Path do if Count > 0 then for i:= 0 to Count - 1 do
    Result:= Result + Path.Strings[i] + '\';
  // formerly just DelimitedText, but that adds "s around items with spaces, so :(
end;

//------------------------------------------------------------------------------

function IsFolder(const Target: String): Boolean;
var s: TSearchRec;
    t: String;
    temp: TStringList;
begin
  {$IFDEF DEBUG}
    Log.Log(['IsFolder("', Target, '"):'], lsNoSpace);
    Log.IncIndentLevel;
  {$ENDIF}
  temp:= TStringList.Create;
  t:= WinBkSlashes(Target);
  if t[Length(t)] = '\' then t:= Copy(t, 1, Length(t) - 1);
  {$IFDEF DEBUG}
    Log.Log(['t = "', t, '";'], lsNoSpace);
  {$ENDIF}
  SplitString(t, '\', temp);
  with temp do Result:=
    (IndexOf('.') = -1)
    and (IndexOf('..') = -1)
    and (AnsiPos('*', t) = 0) and (AnsiPos('?', t) = 0)
    and (FindFirst(t, FolderAttrs, s) = 0) and (s.Attr and faDirectory = faDirectory);
  temp.Free;  
  {$IFDEF DEBUG}
    Log.Log(['Result = ', BoolToStr(Result, True), ';'], lsNoSpace);
    Log.DecIndentLevel;
  {$ENDIF}
end;

//------------------------------------------------------------------------------

procedure FillFilesList(const Spec: String);
var Sr: TSearchRec;
    s, tmp: String; 
begin
  {$IFDEF DEBUG}
    Log.Log(['FillFilesList("', Spec, '");'], lsNoSpace);
    Log.IncIndentLevel;
  {$ENDIF}
  if IsFolder(GetCurPath + Spec) then begin
    Path.Add(WinBkSlashes(Spec));  
    {$IFDEF DEBUG}
      Log.Log('IsFolder = OK;');
      Log.Log(['GetCurPath = "', GetCurPath, '";'], lsNoSpace);
    {$ENDIF}
    s:= GetCurPath + '*';
    if FindFirst(s, FolderAttrs, Sr) = 0 then repeat
      {$IFDEF DEBUG}
        Log.Log(['found item: "', Sr.Name, '";'], lsNoSpace);
      {$ENDIF}
      if (Sr.Attr and faDirectory = faDirectory) then FillFilesList(Sr.Name) else
        Files.Add(GetCurPath + Sr.Name);
    until FindNext(Sr) <> 0;
    FindClose(Sr);
    with Path do Delete(Count - 1); // remove last item
  end;
  {$IFDEF DEBUG}
    Log.DecIndentLevel;
  {$ENDIF}
end;

//------------------------------------------------------------------------------

procedure FindFilesByMask(const Spec: String);
var Sr: TSearchRec;
begin
  if FindFirst(WinBkSlashes(Spec), FileAttrs, Sr) = 0 then repeat
    Files.Add(WorkPath + Sr.Name);
  until FindNext(Sr) <> 0;
  FindClose(Sr);
end;

//------------------------------------------------------------------------------

procedure ProcessList(const ListFileName: String);
var ListContent: TStringList;
    i: Integer;
    RootPath, CurPath, CurMask: String;
    Sr: TSearchRec;
begin
  {$IFDEF DEBUG}
    Log.Log(['ProcessList("', ListFileName, '");'], lsNoSpace);
    Log.IncIndentLevel;
  {$ENDIF}
  ListContent:= TStringList.Create;
  with ListContent do begin
    LoadFromFile(ListFileName);
    i:= 0;
    if Count > 0 then while i < Count do if Strings[i] = EmptyStr then
      Delete(i) else inc(i); // clean empty
    {$IFDEF DEBUG}
      Log.Log(['ListContent.Count = ', IntToStr(Count), ';'], lsNoSpace);
    {$ENDIF}
    if Count > 1 then begin
      RootPath:= WinBkSlashes(Strings[0]);
      if RootPath[Length(RootPath)] <> '\' then RootPath:= RootPath + '\';    
      RootPathLen:= Length(RootPath);
      {$IFDEF DEBUG}
        Log.Log(['RootPath = "', RootPath, '";'], lsNoSpace);
      {$ENDIF}
      for i:= 1 to Count - 1 do begin
        CurMask:= WinBkSlashes(RootPath + Strings[i]);
        CurPath:= ExtractFilePath(CurMask);  
        {$IFDEF DEBUG}
          Log.Log(['CurMask = "', CurMask, '";'], lsNoSpace);
          Log.Log(['Curpath = "', CurPath, '";'], lsNoSpace);
        {$ENDIF}
        if FindFirst(CurMask, FileAttrs, Sr) = 0 then repeat
          {$IFDEF DEBUG}
            Log.Log(['found: ', Sr.Name], lsNoSpace);
          {$ENDIF}
          Files.Add(CurPath + Sr.Name);
        until FindNext(Sr) <> 0;
        FindClose(Sr);
      end;
    end;       
    Free;
  end;
  {$IFDEF DEBUG}
    Log.DecIndentLevel;
  {$ENDIF}
end;

//------------------------------------------------------------------------------

function ArcFileName(const FileName: String): String;
begin
  Result:= CleanString(
    Copy(FileName, RootPathLen + 1, Length(FileName) - RootPathLen)
    , '\', '/');
  {$IFDEF DEBUG}
    Log.Log(['ArcFileName("', FileName, '") = "', Result, '";'], lsNoSpace);
  {$ENDIF}
end;

//==============================================================================
begin
  Writeln('TBZ - utility for tar + bzip2 compression');
  Writeln('Vladimir Slavik 2005, 2006');
  {$IFDEF DEBUG}
    Writeln('Debug version');
    Log:= TLog.Create('tbz_log.log');
    Log.IndentMultiplier:= 2;
  {$ENDIF}

  // prepare temp path
  tempPath:= GetEnvironmentVariable('Temp');
  if tempPath[Length(tempPath)] <> '\' then tempPath:= tempPath + '\';
  (*
  GetLocaleFormatSettings(0, tempFNfmt);
  with tempFNfmt do begin
    DateSeparator:= '_';
    TimeSeparator:= '_';
  end;
  *)
  {$IFDEF DEBUG}
    Log.Log(['tempPath = "', tempPath, '";'], lsNoSpace);
  {$ENDIF}

  // petty items initialization
  FilesSize:= 0;
  GoCompress:= True;
  DriveFree:= DiskFree(Ord(UpperCase(tempPath)[1]) - 64);
  UsedBlockSize:= DefaultBlockSize;
  UsedBufferSize:= DefaultBufferSize;
  Files:= TStringList.Create;
  Path:= TStringList.Create;

  // solve parameters and correct/add paths & extensions as needed
  case ParamCount of
    0, 1, 6..MaxInt: HelpInfo; // invalid # of params
    2: begin // only 2 - op spec and input
      OpSpec:= ParamStr(1);
      InputSpec:= WinBkSlashes(ParamStr(2));
      WorkPath:= ExtractFilePath(InputSpec);
      if FileExists(InputSpec) or IsFolder(InputSpec) then OutputSpec:= WorkPath +
        CleanString(InputSpec, FileNameBadChars, '_') + '.tar.bz2'
        else OutputSpec:= WorkPath + 'Output.tar.bz2';
    end;
    3: begin // switch, op spec and input OR op spec, input, output
      S:= ParamStr(1);
      if Length(S) > 1 then begin // if not, print help and quit
        if S[1] in ['-', '/'] then begin // is a switch?
          case S[2] of
            '1'..'9': UsedBlockSize:= StrToIntDef(S[2], DefaultBlockSize);
            'b': if (Length(S) > 3) and (S[3] = '=') then
              UsedBufferSize:= StrToIntDef(Copy(S, 4, Length(S) - 3),
                DefaultBufferSize);
          end;
          OpSpec:= ParamStr(2);
          InputSpec:= WinBkSlashes(ParamStr(3));
          WorkPath:= ExtractFilePath(InputSpec);
          if FileExists(InputSpec) or IsFolder(InputSpec) then
            OutputSpec:= WorkPath + CleanString(InputSpec, FileNameBadChars, '_')
            + '.tar.bz2'
            else OutputSpec:= WorkPath + 'Output.tar.bz2';
        end else begin // first parameter is not a switch, thus no switches are
          // present and these 3 are task, input, output
          OpSpec:= ParamStr(1);
          InputSpec:= WinBkSlashes(ParamStr(2));
          WorkPath:= ExtractFilePath(InputSpec);
          OutputSpec:= ParamStr(3);
        end;
      end else HelpInfo; // in any case, 1st param can't be shorter than 2 
    end;
    4: begin // two possibilities: 2 switches, opspec, input OR 1 switch, op spec,
      //input, output
      S:= ParamStr(1);
      if Length(S) > 1 then begin
        if S[1] in ['-', '/'] then begin // is a switch?
          case S[2] of // test 1st param
            '1'..'9': UsedBlockSize:= StrToIntDef(S[2], DefaultBlockSize);
            'b': if (Length(S) > 3) and (S[3] = '=') then
              UsedBufferSize:= StrToIntDef(Copy(S, 4, Length(S) - 3),
                DefaultBufferSize);
          end;
          S:= ParamStr(2);
          if Length(S) > 1 then begin // look at 2nd if valid
            if S[1] in ['-', '/'] then begin // is a switch?
              case S[2] of // test 2nd param
                '1'..'9': UsedBlockSize:= StrToIntDef(S[2], DefaultBlockSize);
                'b': if (Length(S) > 3) and (S[3] = '=') then
                  UsedBufferSize:= StrToIntDef(Copy(S, 4, Length(S) - 3),
                  DefaultBufferSize);
              end;
              OpSpec:= ParamStr(3);
              InputSpec:= WinBkSlashes(ParamStr(4));
              WorkPath:= ExtractFilePath(InputSpec);
              if FileExists(InputSpec) or IsFolder(InputSpec) then
                OutputSpec:= WorkPath + CleanString(InputSpec, FileNameBadChars,
                '_') + '.tar.bz2'
                else OutputSpec:= WorkPath + 'Output.tar.bz2';
            end else begin // 2nd is not a switch
              OpSpec:= ParamStr(2);
              InputSpec:= WinBkSlashes(ParamStr(3));
              WorkPath:= ExtractFilePath(InputSpec);
              OutputSpec:= ParamStr(4);
            end;
          end else HelpInfo; // in any case, 2nd param can't be less than 2, too
        end else HelpInfo; // first MUST be a switch here
      end else HelpInfo; // in any case, 1st param can't be shorter than 2 
    end;
    5: begin // both switches, op spec, input, output
      S:= ParamStr(1);
      if Length(S) > 1 then begin // if not, print help and quit
        if S[1] in ['-', '/'] then begin // is a switch?
          case S[2] of
            '1'..'9': UsedBlockSize:= StrToIntDef(S[2], DefaultBlockSize);
            'b': if (Length(S) > 3) and (S[3] = '=') then
              UsedBufferSize:= StrToIntDef(Copy(S, 4, Length(S) - 3),
                DefaultBufferSize);
          end;
          S:= ParamStr(2);
          if Length(S) > 1 then begin
            if S[1] in ['-', '/'] then begin // is a switch?
              case S[2] of // test 2nd param
                '1'..'9': UsedBlockSize:= StrToIntDef(S[2], DefaultBlockSize);
                'b': if (Length(S) > 3) and (S[3] = '=') then
                  UsedBufferSize:= StrToIntDef(Copy(S, 4, Length(S) - 3),
                  DefaultBufferSize);
              end;
              OpSpec:= ParamStr(3);
              InputSpec:= WinBkSlashes(ParamStr(4));
              WorkPath:= ExtractFilePath(InputSpec);
              OutputSpec:= ParamStr(5);
            end else HelpInfo; // 2nd must be a switch, too
          end else HelpInfo; // 2nd must be longer than 1, too
        end else HelpInfo; // 1st must be a switch
      end else HelpInfo; // in any case, 1st param can't be shorter than 2
    end;   
  end; // CASE # of params

  tempFName:= CleanString(
      'TBZ--task-' + OpSpec + '--input-' + InputSpec + '--time-' +
      DateTimeToStr(Now(*, tempFNfmt*)) + '.tar.tmp'
    , FileNameBadChars, '_');

  if (InputSpec[1] = '"') or (InputSpec[Length(InputSpec)-1] = '"') then
    InputSpec:= Copy(InputSpec, 2, Length(inputSpec) - 2);

  {-$DEFINE NoDefaultExt}
  // remove dash to disable extension correction
  {$IFNDEF NoDefaultExt}
    if AnsiPos('.TAR.BZ2', AnsiUpperCase(OutputSpec)) = 0 then
      OutputSpec:= OutputSpec + '.tar.bz2';
  {$ENDIF}

  {$IFDEF DEBUG}
    with Log do begin
      Log(['OpSpec = "', OpSpec, '";'], lsNoSpace);
      Log(['InputSpec = "', InputSpec, '";'], lsNoSpace);
      Log(['OutputSpec = "', OutputSpec, '";'], lsNoSpace);
      Log(['WorkPath = "', WorkPath, '";'], lsNoSpace);
      Log(['tempFName = "', tempFname, '";'], lsNoSpace);
    end;
  {$ENDIF}

  with Files do begin
    CaseSensitive:= False;
    Sorted:= True;
    Duplicates:= dupIgnore;
  end;  

  // do the desired operation
  OpSpec:= UpperCase(OpSpec);
  if OpSpec = 'FILES' then begin
    FindFilesByMask(InputSpec);
    RootPathLen:= 0;
  end else if OpSpec = 'DIR' then begin
    FillFilesList(InputSpec);
    RootPathLen:= Length(InputSpec) + 1;
  end else if OpSpec='LIST' then begin
    ProcessList(InputSpec);
  end else HelpInfo; 
  
  {$IFDEF DEBUG}
    Files.SaveToFile('outputlist.log');
  {$ENDIF}

  if Files.Count > 0 then GoCompress:= True else begin
    GoCompress:= False;
    Writeln(' ! Nothing to compress.')
  end;  

  // test if temp has enough space
  with Files do if GoCompress then begin
    Write('-> Checking for space for temporary tarball ... ');
    for i:= 0 to Count - 1 do begin
      inc(FilesSize, SizeOfFile(Strings[i]));
      if FilesSize + Count * 1024 > DriveFree then begin
        GoCompress:= false;
        Writeln('not enough!');
        Break;
      end;
    end;  
  end;

  // compress files in list:
  // -first add them via Tar to output "Dummy" stream to file in temp
  // -then compress dummy into desired output archive
  // -finally clean temp file and remove stringlist
  if GoCompress and (Files.Count > 0) then begin
    Writeln('OK');
    {$IFDEF DEBUG}
      Log.Log(['GoCompress = True;'], lsNoSpace);
    {$ENDIF}
    Bzip:= TBZip2.Create;
    Bzip.BlockSize:= UsedBlockSize;
    Bzip.BufferSize:= UsedBufferSize;

    Write('-> Opening output stream ... ');
    try
      OutFile:= TFileStream.Create(OutputSpec, fmCreate);
    except
      on E: Exception do Writeln('error');
    end;
    Writeln('OK');
    
    Write('-> Creating tarball ... ');
    X:= GetCurX;
    Y:= GetCurY;
    Writeln;
    temp_Y:= GetCurY;
    try
      Dummy:= TFileStream.Create(tempPath + tempFName, fmCreate);
    except
      on E: Exception do begin
        GoToXY(X, Y);
        Writeln('could not create temporary file!');
      end;
    end;
    Tar:= TTarWriter.Create(Dummy);
    HideCursor;
    for i:= 0 to Files.Count - 1 do begin
      GotoXY(0, temp_y);
      Write(StringOfChar(#32, 160));
      GotoXY(3, temp_y);
      Writeln(Files.Strings[i]);
      Tar.AddFile(Files.Strings[i], ArcFileName(Files.Strings[i]));
    end;
    Tar.Finalize;
    GotoXY(0, temp_y);
    Write(StringOfChar(#32, 160));
    GoToXY(X, Y);
    Writeln('done');
    ShowCursor;

    Write('-> Compressing ... ');
    Bzip.CompressStream(Dummy, OutFile);
    Writeln('done');

    Tar.Free;
    BZip.Free;
    Dummy.Free;
    OutFile.Free;
    if not DeleteFile(tempPath + tempFName) then
      Writeln(' ! Could not delete temporary tarball (', FilesSize,') Bytes.');
  end;
  Files.Free;
  Path.Free;
  Writeln('Finished.');
  {$IFDEF DEBUG}
    Log.Log('Done.');
    Log.Free;
    {$IFDEF CLOSEWAIT}
      Readln;
    {$ENDIF}
  {$ENDIF}
end.
